<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

$page_title = 'Negócios';
$active     = 'negocios';

rbac_require('negocios','view');

$BASE   = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id  = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
$scopeV = rbac_scope('negocios','view'); // 'own' | 'all'

/* ===== CSRF util ===== */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* ===== AJAX: mover card entre etapas ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['ajax'] ?? '') === '1') {
  // Libera o lock da sessão o quanto antes
  session_write_close();
  header('Content-Type: application/json; charset=utf-8');

  try {
    if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) {
      throw new Exception('CSRF inválido');
    }
    if (!rbac_can('negocios','update')) {
      throw new Exception('Sem permissão para atualizar');
    }

    $id    = (int)($_POST['id'] ?? 0);
    $etapa = (string)($_POST['etapa'] ?? '');

    $etapas_ativas = ['novo','qualificando','proposta','negociando'];
    if ($id <= 0 || !in_array($etapa, $etapas_ativas, true)) {
      throw new Exception('Parâmetros inválidos');
    }

    // busca negócio p/ validar escopo
    $neg = fetch_one("SELECT id, owner_user_id, created_by FROM deals WHERE id = ?", [$id], true);
    if (!$neg) { throw new Exception('Negócio não encontrado'); }

    $scopeU = rbac_scope('negocios','update'); // 'own' | 'all'
    if ($scopeU === 'own') {
      $owner = (int)($neg['owner_user_id'] ?? 0);
      $fallback = (int)($neg['created_by'] ?? 0);
      if (!in_array((int)$me_id, [$owner, $fallback], true)) {
        throw new Exception('Sem permissão neste registro');
      }
    }

    // atualiza (com colunas opcionais se existirem)
    try {
      execute("UPDATE deals 
                 SET etapa = ?, stage_changed_at = NOW(), last_activity_at = NOW(), updated_at = NOW() 
               WHERE id = ?", [$etapa, $id]);
    } catch (Throwable $e) {
      execute("UPDATE deals SET etapa = ?, updated_at = NOW() WHERE id = ?", [$etapa, $id]);
    }

    echo json_encode(['ok'=>true]); exit;

  } catch (Throwable $e) {
    http_response_code(400);
    echo json_encode(['ok'=>false, 'msg'=>$e->getMessage()]); exit;
  }
}

/* ===== Helpers ===== */
if (!function_exists('format_money')) {
  function format_money($v){ return 'R$ '.number_format((float)$v, 2, ',', '.'); }
}
if (!function_exists('format_date')) {
  function format_date($dt,$fmt='d/m/Y H:i'){ return $dt ? date($fmt, strtotime($dt)) : '-'; }
}
function badgeEtapa($e) {
  $map = [
    'novo'         => 'secondary',
    'qualificando' => 'info',
    'proposta'     => 'primary',
    'negociando'   => 'warning',
    'ganho'        => 'success',
    'perdido'      => 'danger'
  ];
  return $map[$e] ?? 'secondary';
}
function is_overdue($dt) { return $dt ? (strtotime($dt) < strtotime('today')) : false; }

/* ===== Detecta coluna prob (fallback) ===== */
$hasProb = false;
try { fetch_one("SELECT prob FROM deals LIMIT 1", [], true); $hasProb = true; } catch(Throwable $e){}

/* ===== Abas ===== */
$tab = $_GET['tab'] ?? 'pipeline';
$tabs_validas = ['pipeline','lista','ganhos','perdidos'];
if (!in_array($tab, $tabs_validas, true)) $tab = 'pipeline';

/* ===== Filtros ===== */
$q     = trim($_GET['q'] ?? '');
$de    = $_GET['de']  ?? '';
$ate   = $_GET['ate'] ?? '';
$owner = isset($_GET['owner']) && $_GET['owner'] !== '' ? (int)$_GET['owner'] : null;
if ($scopeV === 'own') { $owner = $me_id ?: null; }

/* ===== Etapas ===== */
$etapas_ativas = ['novo','qualificando','proposta','negociando'];
$etapas_finais = ['ganho','perdido'];
$etapas_todas  = array_merge($etapas_ativas, $etapas_finais);

$etapa_fixa = null;
if ($tab === 'ganhos')   $etapa_fixa = 'ganho';
if ($tab === 'perdidos') $etapa_fixa = 'perdido';

/* ===== WHERE ===== */
$where  = [];
$params = [];

if (!is_null($owner) && $owner > 0) {
  $where[]  = "(d.owner_user_id = ? OR (d.owner_user_id IS NULL AND d.created_by = ?))";
  $params[] = $owner;
  $params[] = $owner;
}
if ($q !== '') {
  $where[]  = "(d.titulo LIKE ? OR c.nome_completo LIKE ?)";
  $params[] = "%{$q}%";
  $params[] = "%{$q}%";
}
if ($de !== '')  { $where[] = "d.data_prevista >= ?"; $params[] = $de; }
if ($ate !== '') { $where[] = "d.data_prevista <= ?"; $params[] = $ate; }
if ($etapa_fixa) { $where[] = "d.etapa = ?";          $params[] = $etapa_fixa; }

$sqlWhere = $where ? ('WHERE '.implode(' AND ', $where)) : '';

/* ===== Owners (somente visão all) ===== */
$owners = [];
if ($scopeV === 'all') {
  $owners = fetch_all("
    SELECT u.id, u.nome
    FROM users u
    JOIN (
      SELECT DISTINCT owner_user_id AS uid FROM deals WHERE owner_user_id IS NOT NULL
      UNION
      SELECT DISTINCT created_by     AS uid FROM deals WHERE owner_user_id IS NULL AND created_by IS NOT NULL
    ) x ON x.uid = u.id
    ORDER BY u.nome ASC
  ", [], true) ?: [];
}

/* ===== KPIs ===== */
$pondExpr = $hasProb ? "COALESCE(SUM(d.valor * COALESCE(d.prob,100)/100),0)" : "COALESCE(SUM(d.valor),0)";
$totais = fetch_one("
  SELECT COUNT(*) AS qtd,
         COALESCE(SUM(d.valor),0) AS total_valor,
         {$pondExpr} AS total_pond
  FROM deals d
  LEFT JOIN clients c ON c.id = d.client_id
  $sqlWhere
", $params, true) ?: ['qtd'=>0,'total_valor'=>0,'total_pond'=>0];

$resumo = fetch_all("
  SELECT d.etapa, COUNT(*) qtd, COALESCE(SUM(d.valor),0) total_valor
  FROM deals d
  LEFT JOIN clients c ON c.id = d.client_id
  $sqlWhere
  GROUP BY d.etapa
", $params, true) ?: [];
$mapResumo = [];
foreach ($resumo as $r) { $mapResumo[$r['etapa']] = $r; }

/* ===== Dados principais ===== */
$rows = fetch_all("
  SELECT d.*,
         ".($hasProb ? "(d.valor * COALESCE(d.prob,100)/100)" : "d.valor")." AS valor_pond,
         c.nome_completo  AS cliente_nome,
         p.titulo         AS imovel_titulo,
         u.nome           AS owner_nome
  FROM deals d
  LEFT JOIN clients    c ON c.id = d.client_id
  LEFT JOIN properties p ON p.id = d.property_id
  LEFT JOIN users      u ON u.id = d.owner_user_id
  $sqlWhere
  ORDER BY d.updated_at DESC, d.id DESC
  LIMIT 300
", $params, true) ?: [];

/* ===== Agrupa p/ pipeline ===== */
$porEtapa = [];
if ($tab === 'pipeline') {
  foreach ($etapas_ativas as $e) $porEtapa[$e] = [];
  foreach ($rows as $r) {
    $e = in_array(($r['etapa'] ?? 'novo'), $etapas_ativas, true) ? $r['etapa'] : 'novo';
    $porEtapa[$e][] = $r;
  }
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
.kpi { padding:.75rem 1rem; border:1px solid rgba(255,255,255,.08); border-radius:.5rem; background:#0e0e10; }
.kpi .label { color:#adb5bd; font-size:.85rem; }
.kpi .value { font-weight:700; font-size:1.05rem; }

.pills .pill { padding:.35rem .6rem; border-radius:1rem; border:1px solid rgba(255,255,255,.1); font-size:.85rem; }
.pills .pill strong { margin-right:.35rem; }

.segment a { border:1px solid rgba(255,255,255,.15); padding:.4rem .7rem; text-decoration:none; color:#e9ecef; }
.segment a.active { background:#1a1a1f; }

.kanban { display:flex; gap:1rem; overflow-x:auto; padding-bottom:.5rem; }
.kan-col { min-width: 300px; background: var(--bs-dark); border:1px solid rgba(255,255,255,.1); border-radius:.5rem; }
.kan-col h6 { padding:.65rem .9rem; margin:0; border-bottom:1px solid rgba(255,255,255,.08); display:flex; justify-content:space-between; align-items:center; }
.kan-list { padding:.7rem; max-height:65vh; overflow:auto; }
.kan-list.drop-target { outline:2px dashed rgba(255,255,255,.25); outline-offset:-6px; }
.kan-card { background:#101114; border:1px solid rgba(255,255,255,.08); border-radius:.5rem; padding:.7rem; margin-bottom:.7rem; transition:transform .08s ease; }
.kan-card.dragging { opacity:.85; transform:scale(.98); }
.kan-card.overdue { border-color: #f7b5b5; box-shadow: 0 0 0 1px rgba(255,0,0,.15) inset; }
.kan-card .top { display:flex; justify-content:space-between; gap:.5rem; }
.kan-card .meta { display:flex; justify-content:space-between; font-size:.85rem; color:#adb5bd; margin-top:.3rem; }
.kan-card .money { font-weight:600; }
.col-count { font-variant-numeric: tabular-nums; }
.toast-lite { position:fixed; top:12px; right:12px; background:#0f5132; color:#d1e7dd; border:1px solid #0f5132; padding:.5rem .75rem; border-radius:.25rem; z-index:1055; display:none; }
.toast-lite.err { background:#58151c; color:#f8d7da; border-color:#58151c; }
</style>

<div class="container my-4">
  <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center gap-2 mb-3">
    <div>
      <div class="d-flex align-items-center gap-2">
        <i class="bi bi-kanban" style="font-size:1.6rem; color: var(--primary);"></i>
        <h1 class="h4 m-0">Negócios</h1>
      </div>
      <div class="text-muted small mt-1"><?= $scopeV==='all' ? 'Visão de equipe' : 'Meus negócios' ?></div>
    </div>
    <div class="d-flex gap-2 align-items-center">
      <div class="segment btn-group">
        <a class="btn btn-sm <?= $tab==='pipeline'?'active':'' ?>" href="<?=$BASE?>/negocios?<?=http_build_query(array_merge($_GET, ['tab'=>'pipeline']))?>">Pipeline</a>
        <a class="btn btn-sm <?= $tab==='lista'?'active':'' ?>"    href="<?=$BASE?>/negocios?<?=http_build_query(array_merge($_GET, ['tab'=>'lista']))?>">Lista</a>
        <a class="btn btn-sm <?= $tab==='ganhos'?'active':'' ?>"   href="<?=$BASE?>/negocios?<?=http_build_query(array_merge($_GET, ['tab'=>'ganhos']))?>">Ganhos</a>
        <a class="btn btn-sm <?= $tab==='perdidos'?'active':'' ?>" href="<?=$BASE?>/negocios?<?=http_build_query(array_merge($_GET, ['tab'=>'perdidos']))?>">Perdidos</a>
      </div>
      <?php if (rbac_can('negocios','create')): ?>
        <a href="<?=$BASE?>/negocios/novo" class="btn btn-primary"><i class="bi bi-plus-lg"></i> Novo</a>
      <?php endif; ?>
    </div>
  </div>

  <!-- Filtros compactos -->
  <form class="card card-body mb-3" method="get">
    <input type="hidden" name="tab" value="<?=htmlspecialchars($tab)?>">
    <div class="row g-2 align-items-end">
      <div class="col-md-4">
        <label class="form-label">Busca</label>
        <input type="text" class="form-control" name="q" value="<?=htmlspecialchars($q)?>" placeholder="Título ou cliente">
      </div>
      <div class="col-md-2">
        <label class="form-label">De</label>
        <input type="date" class="form-control" name="de" value="<?=htmlspecialchars($de)?>">
      </div>
      <div class="col-md-2">
        <label class="form-label">Até</label>
        <input type="date" class="form-control" name="ate" value="<?=htmlspecialchars($ate)?>">
      </div>
      <?php if ($scopeV==='all'): ?>
      <div class="col-md-3">
        <label class="form-label">Responsável</label>
        <select class="form-select" name="owner">
          <option value="">Todos</option>
          <?php foreach ($owners as $o): ?>
            <option value="<?=$o['id']?>" <?=($owner===$o['id'])?'selected':''?>><?=htmlspecialchars($o['nome'])?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <?php endif; ?>
      <div class="col-md-1 d-grid">
        <button class="btn btn-outline-light"><i class="bi bi-funnel"></i></button>
      </div>
    </div>
  </form>

  <!-- KPIs -->
  <div class="d-flex flex-wrap gap-2 mb-3">
    <div class="kpi"><div class="label">Total</div><div class="value"><?=$totais['qtd']??0?></div></div>
    <div class="kpi"><div class="label">Valor</div><div class="value"><?=format_money($totais['total_valor']??0)?></div></div>
    <div class="kpi"><div class="label">Ponderado</div><div class="value"><?=format_money($totais['total_pond']??0)?></div></div>
    <div class="ms-auto pills d-flex gap-2 align-items-center">
      <?php foreach ($etapas_todas as $e):
        $r = $mapResumo[$e] ?? ['qtd'=>0,'total_valor'=>0];
      ?>
        <span class="pill"><strong><?=ucfirst($e)?>:</strong> <?=$r['qtd']?> • <?=format_money($r['total_valor'])?></span>
      <?php endforeach; ?>
    </div>
  </div>

  <?php if ($tab==='lista'): ?>
    <!-- LISTA -->
    <div class="card">
      <div class="table-responsive">
        <table class="table table-dark table-hover align-middle mb-0">
          <thead>
            <tr>
              <th style="width:70px">ID</th>
              <th>Título</th>
              <th>Cliente</th>
              <th>Etapa</th>
              <th style="width:120px">Previsto</th>
              <th class="text-end" style="width:160px">Valor</th>
              <th style="width:160px">Resp.</th>
              <th class="text-end" style="width:180px">Ações</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($rows as $r):
              $canUpd = false; $canDel = false;
              if (rbac_can('negocios','update')) {
                $su = rbac_scope('negocios','update');
                $canUpd = ($su==='all') || ($su==='own' && (int)$r['owner_user_id']===$me_id);
              }
              if (rbac_can('negocios','delete')) {
                $sd = rbac_scope('negocios','delete');
                $canDel = ($sd==='all') || ($sd==='own' && (int)$r['owner_user_id']===$me_id);
              }
            ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td>
                <a class="link-light" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/visualizar"><strong><?=htmlspecialchars($r['titulo'])?></strong></a>
                <?php if (!empty($r['next_action_at'])): ?>
                  <br><small class="<?= is_overdue($r['next_action_at'])?'text-danger':'text-warning' ?>">
                    <i class="bi bi-flag"></i> Próx: <?=format_date($r['next_action_at'], 'd/m H:i')?>
                  </small>
                <?php endif; ?>
                <?php if (!empty($r['updated_at'])): ?>
                  <br><small class="text-muted"><i class="bi bi-clock-history"></i> Atualizado: <?=format_date($r['updated_at'],'d/m H:i')?></small>
                <?php endif; ?>
              </td>
              <td><?= htmlspecialchars($r['cliente_nome'] ?? '-') ?></td>
              <td><span class="badge bg-<?=badgeEtapa($r['etapa'] ?? 'novo')?>"><?=ucfirst($r['etapa'] ?? 'novo')?></span></td>
              <td><?= !empty($r['data_prevista']) ? date('d/m/Y', strtotime($r['data_prevista'])) : '-' ?></td>
              <td class="text-end"><?= format_money($r['valor'] ?? 0) ?></td>
              <td><?= htmlspecialchars($r['owner_nome'] ?? '-') ?></td>
              <td class="text-end">
                <div class="btn-group">
                  <?php if ($canUpd): ?>
                    <a class="btn btn-sm btn-outline-secondary" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/editar"><i class="bi bi-pencil"></i></a>
                  <?php endif; ?>
                  <?php if ($canDel): ?>
                    <a class="btn btn-sm btn-outline-danger" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/remover" onclick="return confirm('Remover este negócio?');"><i class="bi bi-trash"></i></a>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
            <?php endforeach; if (!$rows): ?>
            <tr><td colspan="8" class="text-center text-muted py-4">Nenhum registro</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>

  <?php elseif ($tab==='pipeline'): ?>
    <!-- PIPELINE COM DRAG & DROP (UI OTIMISTA) -->
    <div class="kanban">
      <?php foreach ($etapas_ativas as $e): $cards = $porEtapa[$e] ?? []; ?>
      <div class="kan-col">
        <h6>
          <span><span class="badge bg-<?=badgeEtapa($e)?> me-2">&nbsp;</span><?=ucfirst($e)?></span>
          <small class="text-muted col-count" data-stage="<?=$e?>"><?=count($cards)?></small>
        </h6>
        <div class="kan-list droppable" data-stage="<?=$e?>">
          <?php if (!$cards): ?>
            <div class="text-muted small">Sem itens</div>
          <?php else: foreach ($cards as $r): ?>
            <div class="kan-card <?= !empty($r['next_action_at']) && is_overdue($r['next_action_at']) ? 'overdue' : '' ?>"
                 draggable="true"
                 data-id="<?= (int)$r['id'] ?>"
                 data-stage="<?=$e?>">
              <div class="top">
                <strong><a class="link-light" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/visualizar"><?=htmlspecialchars($r['titulo'])?></a></strong>
                <span class="money"><?=format_money($r['valor'] ?? 0)?></span>
              </div>
              <div class="meta">
                <span class="text-truncate" title="<?=htmlspecialchars($r['cliente_nome'] ?? '-')?>">
                  <i class="bi bi-person"></i> <?= htmlspecialchars($r['cliente_nome'] ?? '-') ?>
                </span>
                <span title="<?=htmlspecialchars($r['owner_nome'] ?? '-')?>">
                  <i class="bi bi-shield"></i> <?= htmlspecialchars($r['owner_nome'] ?? '-') ?>
                </span>
              </div>
              <div class="meta">
                <span><i class="bi bi-calendar-event"></i> <?= $r['data_prevista'] ? date('d/m', strtotime($r['data_prevista'])) : '-' ?></span>
                <?php if (!empty($r['next_action_at'])): ?>
                  <span class="<?= is_overdue($r['next_action_at'])?'text-danger':'' ?>">
                    <i class="bi bi-flag"></i> <?= format_date($r['next_action_at'],'d/m H:i') ?>
                  </span>
                <?php else: ?>
                  <span class="text-muted"><i class="bi bi-flag"></i> —</span>
                <?php endif; ?>
              </div>
              <div class="meta">
                <span class="text-muted"><i class="bi bi-clock-history"></i> Atualizado: <strong class="upd-at"><?= !empty($r['updated_at']) ? format_date($r['updated_at'],'d/m H:i') : '-' ?></strong></span>
                <span>&nbsp;</span>
              </div>
            </div>
          <?php endforeach; endif; ?>
        </div>
      </div>
      <?php endforeach; ?>
    </div>

  <?php else: /* ganhos / perdidos */ ?>
    <div class="card">
      <div class="table-responsive">
        <table class="table table-dark table-hover align-middle mb-0">
          <thead>
            <tr>
              <th style="width:70px">ID</th>
              <th>Título</th>
              <th>Cliente</th>
              <th>Etapa</th>
              <th style="width:120px">Previsto</th>
              <th class="text-end" style="width:160px">Valor</th>
              <th style="width:160px">Resp.</th>
              <th class="text-end" style="width:180px">Ações</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($rows as $r):
              $canUpd = false; $canDel = false;
              if (rbac_can('negocios','update')) {
                $su = rbac_scope('negocios','update');
                $canUpd = ($su==='all') || ($su==='own' && (int)$r['owner_user_id']===$me_id);
              }
              if (rbac_can('negocios','delete')) {
                $sd = rbac_scope('negocios','delete');
                $canDel = ($sd==='all') || ($sd==='own' && (int)$r['owner_user_id']===$me_id);
              }
            ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td>
                <a class="link-light" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/visualizar"><strong><?=htmlspecialchars($r['titulo'])?></strong></a>
                <?php if (($r['etapa'] ?? '')==='perdido' && !empty($r['lost_reason'])): ?>
                  <br><small class="text-danger"><i class="bi bi-x-octagon"></i> <?=htmlspecialchars($r['lost_reason'])?></small>
                <?php endif; ?>
                <?php if (!empty($r['updated_at'])): ?>
                  <br><small class="text-muted"><i class="bi bi-clock-history"></i> Atualizado: <?=format_date($r['updated_at'],'d/m H:i')?></small>
                <?php endif; ?>
              </td>
              <td><?= htmlspecialchars($r['cliente_nome'] ?? '-') ?></td>
              <td><span class="badge bg-<?=badgeEtapa($r['etapa'] ?? 'novo')?>"><?=ucfirst($r['etapa'] ?? 'novo')?></span></td>
              <td><?= !empty($r['data_prevista']) ? date('d/m/Y', strtotime($r['data_prevista'])) : '-' ?></td>
              <td class="text-end"><?= format_money($r['valor'] ?? 0) ?></td>
              <td><?= htmlspecialchars($r['owner_nome'] ?? '-') ?></td>
              <td class="text-end">
                <div class="btn-group">
                  <?php if ($canUpd): ?>
                    <a class="btn btn-sm btn-outline-secondary" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/editar"><i class="bi bi-pencil"></i></a>
                  <?php endif; ?>
                  <?php if ($canDel): ?>
                    <a class="btn btn-sm btn-outline-danger" href="<?=$BASE?>/negocios/<?=(int)$r['id']?>/remover" onclick="return confirm('Remover este negócio?');"><i class="bi bi-trash"></i></a>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
            <?php endforeach; if (!$rows): ?>
            <tr><td colspan="8" class="text-center text-muted py-4">Nenhum registro</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  <?php endif; ?>
</div>

<div id="toast-lite" class="toast-lite" role="status"></div>

<script>
(function(){
  const CSRF = '<?= htmlspecialchars($CSRF) ?>';
  const moveUrl = '<?=$BASE?>/negocios';

  function showToast(msg, err=false){
    const t = document.getElementById('toast-lite');
    if (!t) return;
    t.className = 'toast-lite' + (err?' err':'');
    t.textContent = msg;
    t.style.display = 'block';
    setTimeout(()=>{ t.style.display = 'none'; }, 1200);
  }

  // Drag & Drop – apenas no pipeline
  <?php if ($tab==='pipeline'): ?>
  let dragId = null;
  let dragFromStage = null;

  // throttle de dragover para reduzir reflows
  let raf = null;
  function throttleDragOver(cb){
    return (ev)=>{
      ev.preventDefault();
      if (raf) return;
      raf = requestAnimationFrame(()=>{ raf = null; cb(ev); });
    };
  }

  document.querySelectorAll('.kan-card[draggable="true"]').forEach(card=>{
    card.addEventListener('dragstart', (ev)=>{
      dragId = card.dataset.id;
      dragFromStage = card.dataset.stage;
      card.classList.add('dragging');
      ev.dataTransfer.setData('text/plain', dragId);
      ev.dataTransfer.effectAllowed = 'move';
    });
    card.addEventListener('dragend', ()=>{
      card.classList.remove('dragging');
      dragId = null;
      dragFromStage = null;
      document.querySelectorAll('.droppable').forEach(col=>col.classList.remove('drop-target'));
    });
  });

  document.querySelectorAll('.droppable').forEach(list=>{
    list.addEventListener('dragover', throttleDragOver(()=>{
      list.classList.add('drop-target');
    }));
    list.addEventListener('dragleave', ()=>{
      list.classList.remove('drop-target');
    });
    list.addEventListener('drop', (ev)=>{
      ev.preventDefault();
      list.classList.remove('drop-target');

      const id = ev.dataTransfer.getData('text/plain') || dragId;
      if (!id) return;
      const toStage = list.dataset.stage;
      if (!toStage || toStage === dragFromStage) return;

      // ----- UI OTIMISTA -----
      const card = document.querySelector('.kan-card[data-id="'+id+'"]');
      const fromList = card?.closest('.kan-list');
      if (!card || !fromList) return;

      // move imediatamente
      list.prepend(card);
      card.dataset.stage = toStage;

      // atualiza contadores
      const dec = document.querySelector('.col-count[data-stage="'+(fromList.dataset.stage||'')+'"]');
      const inc = document.querySelector('.col-count[data-stage="'+toStage+'"]');
      if (dec) dec.textContent = Math.max(0, (parseInt(dec.textContent||'0',10)-1));
      if (inc) inc.textContent = (parseInt(inc.textContent||'0',10)+1);

      // seta "Atualizado: agora"
      const upd = card.querySelector('.upd-at');
      if (upd) {
        const d = new Date();
        const dd=('0'+d.getDate()).slice(-2), mm=('0'+(d.getMonth()+1)).slice(-2);
        const hh=('0'+d.getHours()).slice(-2), mi=('0'+d.getMinutes()).slice(-2);
        upd.textContent = `${dd}/${mm} ${hh}:${mi}`;
      }

      // envia em background; reverte se falhar
      const body = new URLSearchParams();
      body.set('ajax','1');
      body.set('csrf', CSRF);
      body.set('id', id);
      body.set('etapa', toStage);

      fetch(moveUrl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: body.toString(),
        keepalive: true
      })
      .then(r=>r.json())
      .then(j=>{
        if (!j.ok) throw new Error(j.msg || 'Falha ao mover');
        showToast('Movido');
      })
      .catch(err=>{
        fromList.prepend(card);
        card.dataset.stage = fromList.dataset.stage || dragFromStage;
        if (dec) dec.textContent = (parseInt(dec.textContent||'0',10)+1);
        if (inc) inc.textContent = Math.max(0, (parseInt(inc.textContent||'0',10)-1));
        showToast(err.message || 'Erro ao mover', true);
      });
    });
  });
  <?php endif; ?>
})();
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>