<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: /troyacrm/login");
    exit;
}

require_once __DIR__ . '/../../app/db.php';

/* === RBAC === */
require_once __DIR__ . '/../../app/rbac.php';
rbac_require('imoveis','view');

$page_title = 'Imóveis';
$BASE = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* ===== Exclusão com log + RBAC ===== */
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    if (!rbac_can('imoveis','delete')) {
        $mensagem_erro = "Você não tem permissão para excluir imóveis.";
    } else {
        try {
            $id = intval($_GET['delete']);

            // Buscar imóvel + possíveis campos de dono para validar escopo
            $imovel_deletado = fetch_one(
                "SELECT id, titulo,
                        /* campos opcionais para escopo */
                        CASE WHEN EXISTS(SELECT 1) THEN owner_user_id END AS owner_user_id,
                        CASE WHEN EXISTS(SELECT 1) THEN created_by END AS created_by
                 FROM properties WHERE id = ?", [$id], true);

            if ($imovel_deletado) {
                $scope = rbac_scope('imoveis','delete'); // 'none' | 'own' | 'all'
                if ($scope === 'own') {
                    $owner_ok = false;
                    if (isset($imovel_deletado['owner_user_id']) && (int)$imovel_deletado['owner_user_id'] === $me_id) {
                        $owner_ok = true;
                    } elseif (isset($imovel_deletado['created_by']) && (int)$imovel_deletado['created_by'] === $me_id) {
                        $owner_ok = true;
                    }
                    if (!$owner_ok) {
                        $mensagem_erro = "Você não tem permissão para excluir este imóvel.";
                    }
                }

                if (empty($mensagem_erro)) {
                    $titulo_deletado = $imovel_deletado['titulo'] ?? ('Imóvel #'.$id);

                    // Registrar log de exclusão ANTES de deletar (se houver tabela de logs)
                    try {
                        $log_gravado = execute(
                            "INSERT INTO property_logs (property_id, user_id, acao, detalhes)
                             VALUES (?, ?, 'excluido', ?)",
                            [$id, $me_id, 'Exclusão do imóvel "' . $titulo_deletado . '"']
                        );

                        // Excluir (independente do log, para não travar operação)
                        execute("DELETE FROM properties WHERE id = ?", [$id]);
                        $mensagem_sucesso = "Imóvel excluído com sucesso!";

                    } catch (Exception $log_error) {
                        error_log("Erro ao gravar log de exclusão: " . $log_error->getMessage());
                        execute("DELETE FROM properties WHERE id = ?", [$id]);
                        $mensagem_sucesso = "Imóvel excluído (log não registrado).";
                    }
                }
            } else {
                $mensagem_erro = "Imóvel não encontrado.";
            }

        } catch (Exception $e) {
            $mensagem_erro = "Erro ao excluir imóvel: " . $e->getMessage();
        }
    }
}

/* ===== Buscar imóveis (aplicar escopo 'own' se necessário) ===== */
$imoveis = [];
$total_imoveis = 0;
$erro = null;

try {
    $sql = "SELECT *
            FROM properties
            ORDER BY created_at DESC
            LIMIT 50";
    $imoveis = fetch_all($sql, [], true);

    // Se o escopo de VIEW for 'own', filtra no PHP usando owner_user_id OU created_by (fallback)
    $scopeView = rbac_scope('imoveis','view'); // 'none' | 'own' | 'all'
    if ($scopeView === 'own') {
        $imoveis = array_values(array_filter($imoveis, function($i) use ($me_id) {
            if (isset($i['owner_user_id'])) return ((int)$i['owner_user_id'] === $me_id);
            if (isset($i['created_by']))   return ((int)$i['created_by']   === $me_id);
            // Se não existir nenhum campo de dono, por segurança não mostra
            return false;
        }));
    }

    $total_imoveis = count($imoveis);
} catch (Exception $e) {
    $erro = $e->getMessage();
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>

<div class="container mt-4 mb-5">
  <div class="d-flex justify-content-between align-items-center mb-4">
    <div>
      <h1 class="mb-1 fw-bold">
        <i class="bi bi-house-door me-2 text-success"></i>
        Imóveis
        <?php if ($total_imoveis > 0): ?>
          <span class="badge bg-secondary"><?php echo $total_imoveis; ?></span>
        <?php endif; ?>
      </h1>
      <p class="text-muted mb-0">Gerencie todos os seus imóveis</p>
    </div>

    <?php if (rbac_can('imoveis','create')): ?>
      <a href="<?=BASE_URL?>/imoveis/novo" class="btn btn-success">
        <i class="bi bi-plus me-2"></i>Novo Imóvel
      </a>
    <?php endif; ?>
  </div>

  <?php if (isset($mensagem_sucesso)): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <i class="bi bi-check-circle me-2"></i><?=$mensagem_sucesso?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if (isset($mensagem_erro)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <i class="bi bi-exclamation-triangle me-2"></i><?=$mensagem_erro?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <?php if ($erro): ?>
    <div class="alert alert-danger">
      <h5><i class="bi bi-exclamation-triangle me-2"></i>Erro ao carregar imóveis</h5>
      <p class="mb-0"><?=htmlspecialchars($erro)?></p>
    </div>
  <?php endif; ?>

  <?php if (empty($imoveis) && !$erro): ?>
    <div class="card">
      <div class="card-body text-center py-5">
        <i class="bi bi-house-door display-1 text-muted mb-3"></i>
        <h3 class="mb-3">Nenhum imóvel cadastrado</h3>
        <p class="text-muted mb-4">Você ainda não cadastrou nenhum imóvel. Comece agora!</p>
        <?php if (rbac_can('imoveis','create')): ?>
          <a href="<?=BASE_URL?>/imoveis/novo" class="btn btn-success btn-lg">
            <i class="bi bi-plus-circle me-2"></i>Cadastrar Primeiro Imóvel
          </a>
        <?php endif; ?>
      </div>
    </div>
  <?php else: ?>
    <div class="card">
      <div class="table-responsive">
        <table class="table table-dark table-hover mb-0 align-middle">
          <thead>
            <tr>
              <th width="60">ID</th>
              <th>Título</th>
              <th width="180">Preço</th>
              <th width="180">Cidade</th>
              <th width="120">Status</th>
              <th width="150" class="text-center">Ações</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($imoveis as $imovel): ?>
              <?php
                // RBAC por linha para ações (update/delete com escopo)
                $canUpdate = false;
                if (rbac_can('imoveis','update')) {
                    $su = rbac_scope('imoveis','update'); // 'own' | 'all'
                    if ($su === 'all') {
                        $canUpdate = true;
                    } else {
                        $owner = $imovel['owner_user_id'] ?? $imovel['created_by'] ?? null;
                        $canUpdate = ($owner !== null && (int)$owner === $me_id);
                    }
                }

                $canDelete = false;
                if (rbac_can('imoveis','delete')) {
                    $sd = rbac_scope('imoveis','delete'); // 'own' | 'all'
                    if ($sd === 'all') {
                        $canDelete = true;
                    } else {
                        $owner = $imovel['owner_user_id'] ?? $imovel['created_by'] ?? null;
                        $canDelete = ($owner !== null && (int)$owner === $me_id);
                    }
                }
              ?>
              <tr>
                <td class="text-center"><?=$imovel['id']?></td>
                <td>
                  <strong><?=htmlspecialchars($imovel['titulo'])?></strong>
                </td>
                <td>
                  <span class="text-success fw-bold"><?=format_money($imovel['preco'])?></span>
                </td>
                <td><?=htmlspecialchars($imovel['cidade'])?></td>
                <td>
                  <?php
                    $status_classes = [
                      'publicado' => 'success',
                      'rascunho' => 'secondary',
                      'vendido' => 'danger',
                      'alugado' => 'info'
                    ];
                    $st = $status_classes[$imovel['status']] ?? 'secondary';
                  ?>
                  <span class="badge bg-<?=$st?>">
                    <?=ucfirst($imovel['status'])?>
                  </span>
                </td>
                <td class="text-center">
                  <div class="btn-group btn-group-sm">
                    <!-- visualizar: precisa ao menos 'view' (já exigimos no topo da página) -->
                    <a href="<?=BASE_URL?>/imoveis/<?=$imovel['id']?>/visualizar" 
                       class="btn btn-outline-success" 
                       title="Visualizar imóvel"
                       target="_blank">
                      <i class="bi bi-eye"></i>
                    </a>

                    <?php if ($canUpdate): ?>
                      <a href="<?=BASE_URL?>/imoveis/<?=$imovel['id']?>/editar" 
                         class="btn btn-outline-primary" 
                         title="Editar imóvel">
                        <i class="bi bi-pencil"></i>
                      </a>
                      <a href="<?=BASE_URL?>/imoveis/<?=$imovel['id']?>/fotos" 
                         class="btn btn-outline-info" 
                         title="Gerenciar fotos">
                        <i class="bi bi-images"></i>
                      </a>
                    <?php endif; ?>

                    <?php if ($canDelete): ?>
                      <a href="<?=BASE_URL?>/imoveis?delete=<?=$imovel['id']?>" 
                         class="btn btn-outline-danger" 
                         title="Excluir imóvel"
                         onclick="return confirm('Tem certeza que deseja excluir este imóvel?\n\nEsta ação não pode ser desfeita!')">
                        <i class="bi bi-trash"></i>
                      </a>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
      <div class="card-footer text-muted d-flex justify-content-between align-items-center">
        <span><i class="bi bi-info-circle me-2"></i>Total: <strong><?=$total_imoveis?></strong> imóve<?=($total_imoveis!=1?'is':'l')?></span>
        <?php
          $publicados = count(array_filter($imoveis, fn($i) => ($i['status'] ?? '') == 'publicado'));
          $rascunhos  = count(array_filter($imoveis, fn($i) => ($i['status'] ?? '') == 'rascunho'));
        ?>
        <div>
          <span class="badge bg-success me-1"><?=$publicados?> publicado<?=$publicados!=1?'s':''?></span>
          <span class="badge bg-secondary"><?=$rascunhos?> rascunho<?=$rascunhos!=1?'s':''?></span>
        </div>
      </div>
    </div>
  <?php endif; ?>
</div>

<?php include __DIR__ . '/../../layout/footer.php'; ?>