<?php
// modules/agenda/calendario.php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

$page_title = 'Agenda - Calendário';
$active     = 'agenda';

rbac_require('agenda','view');

$BASE   = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id  = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
$scopeView = rbac_scope('agenda','view'); // 'own' | 'all'

// Carregar eventos (respeitando escopo)
$params = [];
$sql = "SELECT a.id, a.titulo, a.inicio, a.fim, a.obs, a.owner_user_id,
               p.titulo AS property_titulo,
               c.nome_completo AS cliente_nome
        FROM agenda a
        LEFT JOIN properties p ON p.id = a.property_id
        LEFT JOIN clients    c ON c.id = a.client_id";

if ($scopeView === 'own') {
    $sql .= " WHERE a.owner_user_id = ?";
    $params[] = $me_id;
}
$sql .= " ORDER BY a.inicio DESC LIMIT 500";

$rows = [];
try {
    $rows = fetch_all($sql, $params, true);
} catch (Throwable $e) {
    $rows = [];
}

// Montar array de eventos para o FullCalendar
$events = [];
$can_update = rbac_can('agenda','update');
$upd_scope  = rbac_scope('agenda','update'); // 'own' | 'all'

foreach ($rows as $r) {
    $title_bits = [];
    if (!empty($r['titulo']))           { $title_bits[] = $r['titulo']; }
    if (!empty($r['cliente_nome']))     { $title_bits[] = $r['cliente_nome']; }
    if (!empty($r['property_titulo']))  { $title_bits[] = $r['property_titulo']; }
    $title = implode(' • ', $title_bits) ?: ('#'.$r['id']);

    // URL de clique (editar se permitido, senão apenas lista)
    $allow_edit_this = false;
    if ($can_update) {
        if ($upd_scope === 'all') {
            $allow_edit_this = true;
        } elseif ($upd_scope === 'own' && (int)$r['owner_user_id'] === $me_id) {
            $allow_edit_this = true;
        }
    }
    $url = $allow_edit_this ? ($BASE . '/agenda/editar?id=' . (int)$r['id']) : ($BASE . '/agenda');

    $events[] = [
        'id'    => (int)$r['id'],
        'title' => $title,
        'start' => $r['inicio'],
        'end'   => $r['fim'] ?: null,
        'url'   => $url,
    ];
}

// Injetar JS do calendário via $extra_js (o footer já imprime isso)
$events_json = json_encode($events, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
$extra_js = <<<JS
<script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.11/index.global.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
  const el = document.getElementById('calendar');
  if (!el) return;

  const calendar = new FullCalendar.Calendar(el, {
    locale: 'pt-br',
    initialView: 'dayGridMonth',
    height: 'auto',
    headerToolbar: {
      left: 'prev,next today',
      center: 'title',
      right: 'dayGridMonth,timeGridWeek,timeGridDay,listWeek'
    },
    events: $events_json,
    eventClick: function(info) {
      if (info.event.url) {
        info.jsEvent.preventDefault();
        window.location.href = info.event.url;
      }
    }
  });

  calendar.render();
});
</script>
JS;

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<div class="container my-4">
  <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center gap-2 mb-3">
    <div>
      <div class="d-flex align-items-center gap-2">
        <i class="bi bi-calendar-week" style="font-size:1.6rem; color: var(--success);"></i>
        <h1 class="h4 m-0">Agenda — Calendário</h1>
      </div>
      <div class="text-muted small mt-1">Visualize seus compromissos no calendário</div>
    </div>
    <div class="d-flex gap-2">
      <a href="<?=$BASE?>/agenda" class="btn btn-outline-secondary">
        <i class="bi bi-list-ul"></i> Ver lista
      </a>
      <?php if (rbac_can('agenda','create')): ?>
        <a href="<?=$BASE?>/agenda/novo" class="btn btn-success">
          <i class="bi bi-plus-lg"></i> Novo compromisso
        </a>
      <?php endif; ?>
    </div>
  </div>

  <div class="card">
    <div class="card-body">
      <div id="calendar"></div>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../../layout/footer.php'; ?>