<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

rbac_require('atendimentos','view');

$BASE  = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* ===== Helpers RBAC (fallback, caso não estejam no rbac.php) ===== */
if (!function_exists('rbac_scope_timeline_atendimentos')) {
  function rbac_scope_timeline_atendimentos() {
    if (function_exists('rbac_is_admin') && rbac_is_admin()) return 'all';
    if (function_exists('rbac_acl')) {
      $acl = rbac_acl();
      if (isset($acl['atendimentos']['timeline_add'])) {
        return rbac_scope('atendimentos', 'timeline_add');
      }
    }
    // fallback: update -> view
    $upd = rbac_scope('atendimentos','update');
    if ($upd !== 'none') return $upd;
    return rbac_scope('atendimentos','view');
  }
}
if (!function_exists('rbac_can_timeline_for_atendimento')) {
  function rbac_can_timeline_for_atendimento($row) {
    if (function_exists('rbac_is_admin') && rbac_is_admin()) return true;
    $scope = rbac_scope_timeline_atendimentos(); // 'all' | 'own' | 'none'
    if ($scope === 'all') return true;
    if ($scope === 'own') {
      $me = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
      $owner   = (int)($row['owner_user_id'] ?? 0);
      $creator = (int)($row['created_by'] ?? 0);
      return in_array($me, [$owner, $creator], true);
    }
    return false;
  }
}

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

$allowed_status = ['novo','em_atendimento','negociacao','finalizado'];

if (!function_exists('format_date')) {
  function format_date($dt,$fmt='d/m/Y H:i'){ return $dt ? date($fmt, strtotime($dt)) : '-'; }
}
function badgeStatus($s){
  $map = ['novo'=>'secondary','em_atendimento'=>'info','negociacao'=>'warning','finalizado'=>'success'];
  return $map[$s] ?? 'secondary';
}

/* ----- Carrega atendimento atual (master) ----- */
$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) { header("Location: {$BASE}/atendimentos"); exit; }

try {
  $row = fetch_one("
    SELECT i.*, 
           c.nome_completo AS cliente_nome,
           u.nome          AS owner_nome
    FROM interactions i
    LEFT JOIN clients c ON c.id = i.client_id
    LEFT JOIN users   u ON u.id = COALESCE(i.owner_user_id, i.created_by)
    WHERE i.id = ?
  ", [$id], true);
  if (!$row) { header("Location: {$BASE}/atendimentos"); exit; }

  // Escopo de VIEW
  $scopeV = rbac_scope('atendimentos','view'); // own | all
  if ($scopeV === 'own') {
    $owner = (int)($row['owner_user_id'] ?? $row['created_by'] ?? 0);
    if ($owner !== $me_id) { http_response_code(403); exit('Sem permissão.'); }
  }
} catch (Throwable $e) {
  http_response_code(500); exit('Erro ao carregar.');
}

/* ----- POST: adicionar entrada na timeline (NÃO cria novo interactions) ----- */
$ok_msg = null; $err_msg = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'add_timeline') {
  try {
    if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) {
      throw new Exception('CSRF inválido');
    }

    // Permissão para adicionar timeline (view -> pode lançar timeline no que é seu)
    if (!rbac_can_timeline_for_atendimento($row)) {
      throw new Exception('Sem permissão para adicionar timeline.');
    }

    $status = strtolower(trim($_POST['status'] ?? ''));
    $nota   = trim($_POST['nota'] ?? '');
    $next   = trim($_POST['next_followup_at'] ?? '');

    if (!in_array($status, $allowed_status, true)) {
      throw new Exception('Status inválido.');
    }

    // normaliza datetime-local (YYYY-MM-DDTHH:MM -> Y-m-d H:i:s)
    $next_dt = null;
    if ($next !== '') {
      $next_dt = str_replace('T',' ', substr($next,0,16));
      if (!preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/', $next_dt)) {
        $next_dt = null;
      }
    }

    // 1) Insere LOG (timeline) apontando para o mesmo interaction_id
    execute("
      INSERT INTO interaction_logs (interaction_id, client_id, status, nota, next_followup_at, created_by, created_at)
      VALUES (?, ?, ?, ?, ?, ?, NOW())
    ", [
      (int)$row['id'],
      (int)$row['client_id'],
      $status,
      ($nota !== '' ? $nota : null),
      $next_dt,
      $me_id ?: null
    ]);

    // 2) Atualiza o REGISTRO PRINCIPAL (mantém o mesmo ID)
    execute("
      UPDATE interactions
         SET status = ?, 
             nota = ?,
             next_followup_at = ?,
             updated_at = NOW()
       WHERE id = ?
    ", [
      $status,
      ($nota !== '' ? $nota : null),
      $next_dt,
      (int)$row['id']
    ]);

    // refresca dados locais
    $row['status'] = $status;
    $row['nota']   = ($nota !== '' ? $nota : null);
    $row['next_followup_at'] = $next_dt;
    $row['updated_at'] = date('Y-m-d H:i:s');

    $ok_msg = 'Registro adicionado e atendimento atualizado.';
  } catch (Throwable $e) {
    $err_msg = $e->getMessage();
  }
}

/* ----- Carrega timeline (somente desta interação) ----- */
try {
  $timeline = fetch_all("
    SELECT l.*, u.nome AS owner_nome
      FROM interaction_logs l
 LEFT JOIN users u ON u.id = l.created_by
     WHERE l.interaction_id = ?
  ORDER BY l.created_at DESC, l.id DESC
     LIMIT 100
  ", [$id], true) ?: [];
} catch (Throwable $e) {
  http_response_code(500); exit('Erro ao carregar timeline.');
}

$page_title = 'Atendimento #'.$row['id'];
include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
.timeline { position:relative; }
.timeline:before { content:''; position:absolute; left:10px; top:0; bottom:0; width:2px; background:rgba(255,255,255,.15); }
.tl-item { position:relative; padding-left:32px; margin-bottom:16px; }
.tl-item .dot { position:absolute; left:2px; top:.35rem; width:16px; height:16px; border-radius:50%; background:#6c757d; border:2px solid #111; }
.tl-item .head { display:flex; justify-content:space-between; align-items:center; }
</style>

<div class="container my-4">
  <?php if ($ok_msg): ?>
    <div class="alert alert-success alert-dismissible fade show"><i class="bi bi-check-circle me-2"></i><?=$ok_msg?><button class="btn-close" data-bs-dismiss="alert"></button></div>
  <?php endif; ?>
  <?php if ($err_msg): ?>
    <div class="alert alert-danger alert-dismissible fade show"><i class="bi bi-exclamation-triangle me-2"></i><?=htmlspecialchars($err_msg)?><button class="btn-close" data-bs-dismiss="alert"></button></div>
  <?php endif; ?>

  <div class="card mb-3">
    <div class="card-header d-flex justify-content-between align-items-center">
      <div>
        <strong>Atendimento #<?= (int)$row['id'] ?></strong>
        <span class="badge bg-<?=badgeStatus($row['status'])?> ms-2"><?=ucfirst(str_replace('_',' ',$row['status']))?></span>
      </div>
      <div class="d-flex gap-2">
        <?php if (rbac_can('atendimentos','update')): ?>
          <?php $sU = rbac_scope('atendimentos','update');
                $canEdit = ($sU==='all') || ($sU==='own' && (int)($row['owner_user_id'] ?? $row['created_by']) === $me_id); ?>
          <?php if ($canEdit): ?>
            <a href="<?=$BASE?>/atendimentos/<?= (int)$row['id'] ?>/editar" class="btn btn-sm btn-primary"><i class="bi bi-pencil"></i> Editar</a>
          <?php endif; ?>
        <?php endif; ?>
        <a href="<?=$BASE?>/atendimentos" class="btn btn-sm btn-secondary"><i class="bi bi-arrow-left"></i> Voltar</a>
      </div>
    </div>
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-4">
          <div class="text-muted small">Cliente</div>
          <div class="fw-bold">
            <a class="link-light" href="<?=$BASE?>/clientes/visualizar?id=<?=(int)$row['client_id']?>">
              <?=htmlspecialchars($row['cliente_nome'] ?? '—')?>
            </a>
          </div>
        </div>
        <div class="col-md-3">
          <div class="text-muted small">Responsável</div>
          <div><?=htmlspecialchars($row['owner_nome'] ?? '—')?></div>
        </div>
        <div class="col-md-3">
          <div class="text-muted small">Próxima ação</div>
          <div><?= $row['next_followup_at'] ? format_date($row['next_followup_at']) : '<span class="text-muted">—</span>' ?></div>
        </div>
        <div class="col-md-2">
          <div class="text-muted small">Atualizado em</div>
          <div><?= format_date($row['updated_at'] ?: $row['created_at']) ?></div>
        </div>

        <div class="col-12"><hr></div>

        <div class="col-12">
          <div class="text-muted small mb-1">Nota atual</div>
          <div style="white-space: pre-wrap"><?= $row['nota'] ? htmlspecialchars($row['nota']) : '<span class="text-muted">Sem nota</span>' ?></div>
        </div>
      </div>
    </div>
  </div>

  <?php if (rbac_can_timeline_for_atendimento($row)): ?>
  <!-- FORM RÁPIDO: adicionar item na timeline (log) -->
  <div class="card mb-3">
    <div class="card-header"><strong>Novo registro (timeline)</strong></div>
    <div class="card-body">
      <form method="post" class="row g-3">
        <input type="hidden" name="csrf" value="<?=$CSRF?>">
        <input type="hidden" name="action" value="add_timeline">
        <div class="col-md-3">
          <label class="form-label">Status</label>
          <select name="status" class="form-select" required>
            <?php foreach ($allowed_status as $s): ?>
              <option value="<?=$s?>" <?=$row['status']===$s?'selected':''?>><?=ucfirst(str_replace('_',' ',$s))?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-md-4">
          <label class="form-label">Próxima ação</label>
          <input type="datetime-local" name="next_followup_at" class="form-control">
        </div>
        <div class="col-12">
          <label class="form-label">Nota</label>
          <textarea name="nota" rows="3" class="form-control" placeholder="Descreva a interação, ligação, e-mail, retorno..."></textarea>
        </div>
        <div class="col-12 d-flex gap-2">
          <button class="btn btn-success"><i class="bi bi-plus-lg me-1"></i>Adicionar à timeline</button>
          <a class="btn btn-outline-secondary" href="<?=$BASE?>/atendimentos/<?=$id?>/visualizar">Limpar</a>
        </div>
      </form>
    </div>
  </div>
  <?php endif; ?>

  <!-- TIMELINE (somente desta interação) -->
  <div class="card">
    <div class="card-header">
      <strong>Linha do Tempo</strong>
      <small class="text-muted ms-2">(Cliente: <?=htmlspecialchars($row['cliente_nome'] ?? '—')?>)</small>
    </div>
    <div class="card-body">
      <?php if (!$timeline): ?>
        <div class="text-muted">Sem eventos ainda.</div>
      <?php else: ?>
        <div class="timeline">
          <?php foreach ($timeline as $t): ?>
            <div class="tl-item">
              <span class="dot" style="background: var(--bs-<?=badgeStatus($t['status'])?>);"></span>
              <div class="head">
                <div>
                  <span class="badge bg-<?=badgeStatus($t['status'])?>"><?=ucfirst(str_replace('_',' ',$t['status']))?></span>
                  <small class="text-muted ms-2"><?= format_date($t['created_at']) ?></small>
                </div>
                <small class="text-muted"><i class="bi bi-person"></i> <?= htmlspecialchars($t['owner_nome'] ?? '—') ?></small>
              </div>
              <?php if (!empty($t['nota'])): ?>
              <div class="mt-1" style="white-space: pre-wrap"><?= htmlspecialchars($t['nota']) ?></div>
              <?php endif; ?>
              <?php if (!empty($t['next_followup_at'])): ?>
              <div class="mt-1 text-warning"><i class="bi bi-flag"></i> Próx. ação: <?= format_date($t['next_followup_at']) ?></div>
              <?php endif; ?>
            </div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../../layout/footer.php'; ?>