<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: /troyacrm/login");
    exit;
}

require_once __DIR__ . '/../../app/db.php';

/* === RBAC: exigir permissão para criar clientes === */
require_once __DIR__ . '/../../app/rbac.php';
rbac_require('clientes','create');

$page_title = 'Novo Cliente';

$erros = [];
$sucesso = null;

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nome_completo = trim($_POST['nome_completo'] ?? '');
    $cpf_cnpj = trim($_POST['cpf_cnpj'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $telefone_principal = trim($_POST['telefone_principal'] ?? '');
    $telefone_secundario = trim($_POST['telefone_secundario'] ?? '');
    $tipo_cliente = $_POST['tipo_cliente'] ?? 'comprador';
    $endereco = trim($_POST['endereco'] ?? '');
    $bairro = trim($_POST['bairro'] ?? '');
    $cidade = trim($_POST['cidade'] ?? '');
    $uf = trim($_POST['uf'] ?? '');
    $cep = trim($_POST['cep'] ?? '');
    $observacoes = trim($_POST['observacoes'] ?? '');
    $data_nascimento = $_POST['data_nascimento'] ?? '';

    // Validações
    if (empty($nome_completo)) {
        $erros[] = "O nome completo é obrigatório.";
    }

    if (empty($telefone_principal)) {
        $erros[] = "O telefone principal é obrigatório.";
    }

    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $erros[] = "E-mail inválido.";
    }

    if (empty($erros)) {
        try {
            $resultado = execute(
                "INSERT INTO clients (nome_completo, cpf_cnpj, email, telefone_principal, telefone_secundario, tipo_cliente, endereco, bairro, cidade, uf, cep, observacoes, data_nascimento, created_by) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                [
                    $nome_completo,
                    $cpf_cnpj ?: null,
                    $email ?: null,
                    $telefone_principal,
                    $telefone_secundario ?: null,
                    $tipo_cliente,
                    $endereco ?: null,
                    $bairro ?: null,
                    $cidade ?: null,
                    $uf ?: null,
                    $cep ?: null,
                    $observacoes ?: null,
                    $data_nascimento ?: null,
                    $_SESSION['user_id'] ?? null
                ]
            );

            if ($resultado) {
                header("Location: " . BASE_URL . "/clientes?success=1");
                exit;
            } else {
                $erros[] = "Erro ao cadastrar cliente. Tente novamente.";
            }
        } catch (Exception $e) {
            $erros[] = "Erro no banco de dados: " . $e->getMessage();
        }
    }
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>

<div class="container mt-4 mb-5">
  <div class="row justify-content-center">
    <div class="col-lg-8">
      <div class="card">
        <div class="card-header bg-primary text-white">
          <h4 class="mb-0">
            <i class="bi bi-person-plus me-2"></i>Novo Cliente
          </h4>
        </div>
        <div class="card-body">
          <?php if (!empty($erros)): ?>
            <div class="alert alert-danger">
              <h6><i class="bi bi-exclamation-triangle me-2"></i>Erros encontrados:</h6>
              <ul class="mb-0">
                <?php foreach ($erros as $erro): ?>
                  <li><?=htmlspecialchars($erro)?></li>
                <?php endforeach; ?>
              </ul>
            </div>
          <?php endif; ?>

          <form method="POST" novalidate>
            <div class="row">
              <!-- Dados principais -->
              <div class="col-md-6 mb-3">
                <label for="nome_completo" class="form-label">Nome Completo *</label>
                <input type="text" class="form-control" id="nome_completo" name="nome_completo" 
                       value="<?=htmlspecialchars($_POST['nome_completo'] ?? '')?>" required 
                       maxlength="255" placeholder="João Silva">
                <div class="form-text">Nome completo do cliente</div>
              </div>

              <div class="col-md-6 mb-3">
                <label for="cpf_cnpj" class="form-label">CPF/CNPJ</label>
                <input type="text" class="form-control" id="cpf_cnpj" name="cpf_cnpj" 
                       value="<?=htmlspecialchars($_POST['cpf_cnpj'] ?? '')?>" 
                       maxlength="20" placeholder="000.000.000-00 ou 00.000.000/0000-00">
                <div class="form-text">Opcional - para identificação fiscal</div>
              </div>

              <div class="col-md-6 mb-3">
                <label for="email" class="form-label">E-mail</label>
                <input type="email" class="form-control" id="email" name="email" 
                       value="<?=htmlspecialchars($_POST['email'] ?? '')?>" 
                       placeholder="joao@email.com">
                <div class="form-text">Opcional - para contato</div>
              </div>

              <div class="col-md-6 mb-3">
                <label for="telefone_principal" class="form-label">Telefone Principal *</label>
                <input type="tel" class="form-control" id="telefone_principal" name="telefone_principal" 
                       value="<?=htmlspecialchars($_POST['telefone_principal'] ?? '')?>" required 
                       maxlength="20" placeholder="(41) 99999-9999">
                <div class="form-text">Telefone ou WhatsApp principal</div>
              </div>

              <div class="col-md-6 mb-3">
                <label for="telefone_secundario" class="form-label">Telefone Secundário</label>
                <input type="tel" class="form-control" id="telefone_secundario" name="telefone_secundario" 
                       value="<?=htmlspecialchars($_POST['telefone_secundario'] ?? '')?>" 
                       maxlength="20" placeholder="(41) 99999-9998">
                <div class="form-text">Opcional - telefone adicional</div>
              </div>

              <div class="col-md-6 mb-3">
                <label for="tipo_cliente" class="form-label">Tipo de Cliente *</label>
                <select class="form-select" id="tipo_cliente" name="tipo_cliente" required>
                  <option value="comprador" <?=($_POST['tipo_cliente'] ?? 'comprador') == 'comprador' ? 'selected' : ''?>>
                    Comprador
                  </option>
                  <option value="vendedor" <?=($_POST['tipo_cliente'] ?? '') == 'vendedor' ? 'selected' : ''?>>
                    Vendedor
                  </option>
                  <option value="locatario" <?=($_POST['tipo_cliente'] ?? '') == 'locatario' ? 'selected' : ''?>>
                    Locatário
                  </option>
                  <option value="proprietario" <?=($_POST['tipo_cliente'] ?? '') == 'proprietario' ? 'selected' : ''?>>
                    Proprietário
                  </option>
                </select>
              </div>

              <div class="col-md-6 mb-3">
                <label for="data_nascimento" class="form-label">Data de Nascimento</label>
                <input type="date" class="form-control" id="data_nascimento" name="data_nascimento" 
                       value="<?=htmlspecialchars($_POST['data_nascimento'] ?? '')?>">
                <div class="form-text">Opcional - para análise de perfil</div>
              </div>

              <!-- Endereço -->
              <div class="col-12 mb-4">
                <h6 class="text-primary border-start border-3 border-primary ps-3">
                  <i class="bi bi-geo-alt me-2"></i>Endereço
                </h6>
              </div>

              <div class="col-md-4 mb-3">
                <label for="cep" class="form-label">CEP</label>
                <input type="text" class="form-control" id="cep" name="cep" 
                       value="<?=htmlspecialchars($_POST['cep'] ?? '')?>" 
                       maxlength="10" placeholder="00000-000">
              </div>

              <div class="col-md-8 mb-3">
                <label for="endereco" class="form-label">Endereço</label>
                <input type="text" class="form-control" id="endereco" name="endereco" 
                       value="<?=htmlspecialchars($_POST['endereco'] ?? '')?>" 
                       placeholder="Rua das Flores, 123">
              </div>

              <div class="col-md-4 mb-3">
                <label for="bairro" class="form-label">Bairro</label>
                <input type="text" class="form-control" id="bairro" name="bairro" 
                       value="<?=htmlspecialchars($_POST['bairro'] ?? '')?>" 
                       placeholder="Centro">
              </div>

              <div class="col-md-5 mb-3">
                <label for="cidade" class="form-label">Cidade</label>
                <input type="text" class="form-control" id="cidade" name="cidade" 
                       value="<?=htmlspecialchars($_POST['cidade'] ?? '')?>" 
                       placeholder="Curitiba">
              </div>

              <div class="col-md-3 mb-3">
                <label for="uf" class="form-label">UF</label>
                <select class="form-select" id="uf" name="uf">
                  <option value="">Selecione...</option>
                  <?php
                  $ufs = ['AC','AL','AP','AM','BA','CE','DF','ES','GO','MA','MT','MS','MG','PA','PB','PR','PE','PI','RJ','RN','RS','RO','RR','SC','SP','SE','TO'];
                  foreach ($ufs as $uf_option) {
                      $selected = ($_POST['uf'] ?? '') == $uf_option ? 'selected' : '';
                      echo "<option value='$uf_option' $selected>$uf_option</option>";
                  }
                  ?>
                </select>
              </div>

              <!-- Observações -->
              <div class="col-12 mb-4">
                <h6 class="text-primary border-start border-3 border-primary ps-3">
                  <i class="bi bi-chat-text me-2"></i>Observações
                </h6>
              </div>

              <div class="col-12 mb-3">
                <label for="observacoes" class="form-label">Observações</label>
                <textarea class="form-control" id="observacoes" name="observacoes" rows="4" 
                          placeholder="Observações sobre o cliente, preferências, histórico..."><?=htmlspecialchars($_POST['observacoes'] ?? '')?></textarea>
                <div class="form-text">Opcional - informações adicionais sobre o cliente</div>
              </div>
            </div>

            <div class="d-grid gap-2 d-md-flex justify-content-md-end">
              <a href="<?=BASE_URL?>/clientes" class="btn btn-secondary me-md-2">
                <i class="bi bi-arrow-left me-2"></i>Voltar
              </a>
              <button type="submit" class="btn btn-primary">
                <i class="bi bi-check-circle me-2"></i>Salvar Cliente
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  </div>
</div>

<script>
// Máscara para CPF/CNPJ
document.addEventListener('DOMContentLoaded', function() {
    const cpfCnpjInput = document.getElementById('cpf_cnpj');
    const telefonePrincipal = document.getElementById('telefone_principal');
    const telefoneSecundario = document.getElementById('telefone_secundario');
    const cepInput = document.getElementById('cep');

    if (cpfCnpjInput) {
        cpfCnpjInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            if (value.length <= 11) {
                // CPF
                value = value.replace(/(\d{3})(\d)/, '$1.$2');
                value = value.replace(/(\d{3})(\d)/, '$1.$2');
                value = value.replace(/(\d{3})(\d{1,2})$/, '$1-$2');
            } else {
                // CNPJ
                value = value.replace(/^(\d{2})(\d)/, '$1.$2');
                value = value.replace(/^(\d{2}\.\d{3})(\d)/, '$1.$2');
                value = value.replace(/^(\d{2}\.\d{3}\.\d{3})(\d)/, '$1/$2');
                value = value.replace(/^(\d{2}\.\d{3}\.\d{3}\/\d{4})(\d{2})$/, '$1-$2');
            }
            e.target.value = value;
        });
    }

    // Máscara para telefones
    function applyTelefoneMask(input) {
        input.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            value = value.replace(/(\d{2})(\d)/, '($1) $2');
            value = value.replace(/(\d{5})(\d)/, '$1-$2');
            value = value.replace(/(\d{4})(\d)/, '$1$2');
            e.target.value = value.substring(0, 15);
        });
    }

    if (telefonePrincipal) applyTelefoneMask(telefonePrincipal);
    if (telefoneSecundario) applyTelefoneMask(telefoneSecundario);

    // Máscara para CEP
    if (cepInput) {
        cepInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            value = value.replace(/(\d{5})(\d)/, '$1-$2');
            e.target.value = value.substring(0, 9);
        });
    }
});
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>