<?php
// modules/imoveis/thumb.php
// Gera thumbnail "cover" via GD (com cache em disco).
// Uso: /troyacrm/modules/imoveis/thumb.php?pid=123&f=arquivo.jpg&w=900&h=600

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

require_once __DIR__ . '/../../app/db.php';   // p/ garantir autoload de config/db
require_once __DIR__ . '/../../app/rbac.php'; // pode não ser usado se acesso interno

// ---------- Parâmetros ----------
$pid = (int)($_GET['pid'] ?? 0);
$fn  = basename((string)($_GET['f'] ?? ''));
$w   = max(16, min(2400, (int)($_GET['w'] ?? 900)));
$h   = max(16, min(2400, (int)($_GET['h'] ?? 600)));

if ($pid <= 0 || $fn === '') { http_response_code(400); exit('Parâmetros inválidos'); }

// ---------- Autorização ----------
// 1) Se logado: ok
$logged = !empty($_SESSION['logged_in']);

// 2) Se chamada interna (servidor chamando a si mesmo) também permite
$remote = $_SERVER['REMOTE_ADDR'] ?? '';
$server = $_SERVER['SERVER_ADDR'] ?? '';
$internal = in_array($remote, ['127.0.0.1','::1',$server], true);

// 3) Opcional: token assinado (se quiser usar no futuro)
// $config = require __DIR__ . '/../../config/config.php';
// $secret = $config['app']['secret'] ?? 'change-me';
// $k = $_GET['k'] ?? '';
// $validToken = hash_equals(hash_hmac('sha256', "$pid|$fn|$w|$h", $secret), $k);

if (!$logged && !$internal /* && !$validToken */) {
  // Se quiser endurecer, troque por 403.
  // Aqui mando 200 mas com fallback de blur para não vazar nada sensível.
  // Para simplificar: vamos só negar.
  http_response_code(403); exit('Acesso negado');
}

// ---------- Arquivos / Cache ----------
$srcPath   = __DIR__ . "/../../uploads/imoveis/{$pid}/{$fn}";
if (!is_file($srcPath)) { http_response_code(404); exit('Arquivo não encontrado'); }

$cacheDir  = __DIR__ . "/../../uploads/imoveis/_thumbs/{$pid}";
$cacheFile = "{$cacheDir}/{$w}x{$h}-{$fn}.jpg";
if (!is_dir($cacheDir)) { @mkdir($cacheDir, 0755, true); }

// Se thumb em cache for mais nova que a origem, serve do cache
$srcMTime = filemtime($srcPath) ?: time();
$useCache = is_file($cacheFile) && (filemtime($cacheFile) >= $srcMTime);

// Headers de cache HTTP
$etag = sprintf('W/"%s-%dx%d-%s"', md5($fn.$srcMTime.$w.$h), $w, $h, $pid);
header('ETag: '.$etag);
header('Cache-Control: public, max-age=31536000, immutable');
header('Last-Modified: '.gmdate('D, d M Y H:i:s', ($useCache ? filemtime($cacheFile) : $srcMTime)).' GMT');
if ((isset($_SERVER['HTTP_IF_NONE_MATCH']) && trim($_SERVER['HTTP_IF_NONE_MATCH']) === $etag)) {
  http_response_code(304); exit;
}

if ($useCache) {
  header('Content-Type: image/jpeg');
  readfile($cacheFile);
  exit;
}

// ---------- GD / Resize (cover) ----------
$info = @getimagesize($srcPath);
if (!$info) { http_response_code(415); exit('Imagem inválida'); }
$mime = $info['mime'] ?? 'image/jpeg';

switch ($mime) {
  case 'image/jpeg': $src = @imagecreatefromjpeg($srcPath); break;
  case 'image/png':  $src = @imagecreatefrompng($srcPath);  break;
  case 'image/webp': $src = function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($srcPath) : null; break;
  default:
    $blob = @file_get_contents($srcPath);
    $src  = $blob ? @imagecreatefromstring($blob) : null;
}
if (!$src) { http_response_code(415); exit('Falha ao abrir imagem'); }

$iw = imagesx($src);
$ih = imagesy($src);

$ratioBox = $w / $h;
$ratioImg = $iw / $ih;

// Calcula recorte (cover)
if ($ratioImg > $ratioBox) {
  $sh = $ih;
  $sw = (int)round($ih * $ratioBox);
  $sx = (int)round(($iw - $sw) / 2);
  $sy = 0;
} else {
  $sw = $iw;
  $sh = (int)round($iw / $ratioBox);
  $sx = 0;
  $sy = (int)round(($ih - $sh) / 2);
}

$dst = imagecreatetruecolor($w, $h);
$bg  = imagecolorallocate($dst, 0, 0, 0);
imagefilledrectangle($dst, 0, 0, $w, $h, $bg);
imageinterlace($dst, true);
imagealphablending($dst, true);
imagesavealpha($dst, true);
imagecopyresampled($dst, $src, 0, 0, $sx, $sy, $w, $h, $sw, $sh);

// Salva em cache + envia
header('Content-Type: image/jpeg');
@imagejpeg($dst, $cacheFile, 80); // grava cache
imagejpeg($dst, null, 80);        // envia

imagedestroy($dst);
imagedestroy($src);