<?php
// modules/imoveis/exportar.php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';
rbac_require('imoveis','view');

$BASE = defined('BASE_URL') ? BASE_URL : '/troyacrm';

/* ==== Obter ID ==== */
$property_id = (int)($_GET['id'] ?? 0);
if ($property_id <= 0) {
  $uri = $_SERVER['REQUEST_URI'] ?? '';
  if (preg_match('~\/imoveis\/([0-9]+)\/exportar~', $uri, $m)) {
    $property_id = (int)$m[1];
  }
}
if ($property_id <= 0) { http_response_code(400); echo 'Imóvel inválido (ID ausente).'; exit; }

/* ==== Carregar imóvel ==== */
try {
  $imovel = fetch_one("SELECT * FROM properties WHERE id = ? LIMIT 1", [$property_id], true);
  if (!$imovel) { throw new Exception('not found'); }
} catch (Throwable $e) {
  http_response_code(500); echo 'Erro ao carregar imóvel.'; exit;
}

/* ==== Condomínio (opcional) ==== */
$condominio_nome = '';
try {
  if (!empty($imovel['condominio_id'])) {
    $c = fetch_one("SELECT nome FROM condominios WHERE id = ?", [(int)$imovel['condominio_id']], true);
    $condominio_nome = $c['nome'] ?? '';
  }
} catch (Throwable $e) {}

/* ==== Características (chips) ==== */
$caracts = [];
try {
  $caracts = fetch_all("
    SELECT pf.nome, pf.categoria
      FROM property_feature_values v
      JOIN property_features pf ON pf.id = v.feature_id
     WHERE v.property_id = ?
     ORDER BY pf.categoria, pf.nome
  ", [$property_id], true) ?: [];
} catch (Throwable $e) {}

/* ==== Fotos ==== */
try {
  $fotos = fetch_all("
    SELECT id, filename, path, COALESCE(ordem,0) AS ordem, created_at
      FROM property_photos
     WHERE property_id = ?
     ORDER BY COALESCE(ordem,0) ASC, created_at ASC, id ASC
  ", [$property_id], true) ?: [];
} catch (Throwable $e) { $fotos = []; }

/* ==== Helpers ==== */
$upload_dir_web = rtrim($BASE, '/') . '/uploads/imoveis/' . $property_id . '/';
$foto_url = function($row) use ($upload_dir_web) {
  if (!empty($row['path'])) return $row['path'];
  return $upload_dir_web . $row['filename'];
};
function e($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function v($arr, $k, $def='—'){ return (isset($arr[$k]) && $arr[$k] !== '' && $arr[$k] !== null) ? $arr[$k] : $def; }
function fmt_money($v){ return ($v===null||$v==='')?'—':'R$ '.number_format((float)$v,2,',','.'); }

/* ==== Campos base ==== */
$title  = (string)($imovel['titulo'] ?? ('Imóvel #'.$property_id));
$bairro = (string)($imovel['bairro'] ?? '');
$cidade = (string)($imovel['cidade'] ?? '');
$uf     = (string)($imovel['uf'] ?? '');
$sub    = trim(($bairro ? $bairro.' · ' : '') . $cidade . ($uf ? '/'.$uf : ''));
$hero   = $fotos[0] ?? null;

/* ==== Logo (qualidade) ==== */
$DOCROOT = rtrim($_SERVER['DOCUMENT_ROOT'] ?? '', '/');
$logo_base_web = $BASE . '/assets/img/troya-imoveis-logo';
$logo_url = $logo_base_web . '.svg';
if (!is_file($DOCROOT . $logo_url)) {
  $logo_url = $logo_base_web . '.png';
}
$logo_srcset = '';
$src2x = $logo_base_web . '@2x.png';
$src3x = $logo_base_web . '@3x.png';
$parts = [];
if (is_file($DOCROOT . $src2x)) { $parts[] = e($src2x) . ' 2x'; }
if (is_file($DOCROOT . $src3x)) { $parts[] = e($src3x) . ' 3x'; }
if ($parts) { $logo_srcset = implode(', ', $parts); }
?>
<!doctype html>
<html lang="pt-br">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Exportar PDF — <?= e($title) ?></title>
<style>
  :root{
    --pw: 595.28pt;         /* A4 W */
    --ph: 841.89pt;         /* A4 H */
    --m: 24pt;              /* margem interna */
    --gap: 10pt;
    --footerH: 18pt;

    /* Altura do HERO — cabe no A4 junto com blocos */
    --heroH: 210pt;

    --muted: #6b7280;
    --chipBg:#f3f4f6;
    --chipBd:#e5e7eb;
  }
  body{ margin:0; background:#0d1117; color:#e6edf3; font-family: system-ui, -apple-system, Segoe UI, Roboto, Ubuntu, "Helvetica Neue", Arial, "Noto Sans", sans-serif; }
  .wrap{ max-width:900px; margin:12px auto; padding:8px; }
  .hint{ margin:8px 0 12px; color:#9aa4b2; font-size:14px; }
  .progress{ height:10px; background:#0f2533; border-radius:999px; overflow:hidden; margin:8px 0 16px; border:1px solid rgba(255,255,255,.08); }
  .bar{ height:100%; width:0%; background:#2ea043; transition:width .15s linear; }
  .btns a{ color:#9ecbff; text-decoration:none; font-size:14px; }

  .sheet{ width: var(--pw); height: var(--ph); box-sizing:border-box; background:#fff; color:#111;
          margin: 12px auto; box-shadow: 0 0 0 1px rgba(255,255,255,.06), 0 8px 24px rgba(0,0,0,.35); position: relative; }
  .sheet .inner{ padding: var(--m); padding-bottom: calc(var(--m) + var(--footerH) + 6pt); }

  /* CAPA */
  .topbar{ display:flex; align-items:center; justify-content:space-between; }
  .logo{ height: 30pt; }
  .right-top{ color:#6b7280; font-size:10pt; font-weight:700; letter-spacing:.4px; }

  .hero{
    height: var(--heroH);
    border-radius: 8pt; overflow:hidden; position:relative; margin: 12pt 0;
    background-color:#e5e7eb;
    background-position:center; background-size:cover; background-repeat:no-repeat; /* evita esticar */
  }
  .hero::after{ content:''; position:absolute; inset:0;
    background: linear-gradient(180deg, rgba(0,0,0,.08), rgba(0,0,0,.16)); }
  .hero .stripe{
    position:absolute; left:0; right:0; bottom:0; height: 28pt;
    background: rgba(0,0,0,.60); display:flex; align-items:center; padding: 0 10pt;
  }
  .hero .h-title{ color:#fff; font-weight:800; font-size:13.5pt; margin-right:12pt; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
  .hero .h-price{ color:#fff; font-size:12pt; opacity:.95; }

  .title{ font-weight:800; font-size: 21pt; line-height:1.2; margin: 2pt 0 2pt; }
  .subtitle{ color: var(--muted); font-size: 10.5pt; margin-bottom: 8pt; }

  /* chips principais abaixo do hero */
  .chips{ display:flex; flex-wrap:wrap; gap:6pt; }
  .chip{ background: var(--chipBg); border:1px solid var(--chipBd); border-radius:999px; padding:4pt 10pt; font-size:9.8pt; }

  /* faixa de specs compacta (6 itens) */
  .specs{
    display:grid; grid-template-columns: repeat(3, 1fr); gap: 8pt 12pt;
    border:1px solid #e5e7eb; border-radius:8pt; padding:10pt 12pt; background:#fafafa; margin-top:10pt;
  }
  .kv .label{ color: var(--muted); font-size:9.5pt; }
  .kv .value{ font-weight:700; font-size:12pt; margin-top:2pt; }

  /* badges contadores */
  .badges{ display:flex; flex-wrap:wrap; gap:6pt; margin-top:10pt; }
  .badge{ background: var(--chipBg); border:1px solid var(--chipBd); border-radius:999px; padding:4pt 10pt; font-size:10pt; }

  .section-title{ font-weight:700; margin-top:12pt; margin-bottom:6pt; font-size:11pt; }
  .desc-title{ margin-top:10pt; font-weight:700; font-size:11pt; }
  .desc{
    margin-top:6pt;
    font-size:10.5pt;
    color:#374151;
    text-align: justify;
  }

  .footer{ position:absolute; left: var(--m); right: var(--m); bottom: var(--m); display:flex; justify-content:space-between; color:#6b7280; font-size:9pt; }

  /* FOTOS: 2 por página – usando background cover para não achatar */
  .inner.photos{ padding: calc(var(--m) - 4pt) var(--m) calc(var(--m) + var(--footerH)); }
  .photo-page{ display:flex; flex-direction:column; gap: var(--gap); }
  .frame{
    width: 100%;
    /* altura calculada para caber 2 por página sem amassar */
    height: calc((var(--ph) - (var(--m) * 2) - var(--footerH) - var(--gap)) / 2);
    border:1px solid #e5e7eb; background:#f8fafc;
    position:relative; overflow:hidden; border-radius: 6pt;
    background-position:center; background-size:cover; background-repeat:no-repeat; /* sem esticar */
  }
  .legend{ position:absolute; top:6pt; left:8pt; color:#374151; font-size:9.5pt; background:#fff8; padding:2pt 6pt; border-radius:6pt; border:1px solid #e5e7eb; }

  .actions{ margin:10px auto; max-width:900px; display:flex; gap:8px; }
  .btn{ display:inline-block; background:#238636; color:#fff; border:none; border-radius:6px; padding:8px 12px; cursor:pointer; }
  .btn.secondary{ background:#30363d; }
</style>
</head>
<body>
<div class="wrap">
  <div class="hint">Gerando o PDF de <strong><?= e($title) ?></strong>… o download iniciará automaticamente.</div>
  <div class="progress"><div id="bar" class="bar"></div></div>
  <div class="btns"><a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/fotos">Voltar às fotos</a></div>
</div>

<!-- CAPA -->
<div class="sheet" data-sheet="capa">
  <div class="inner">
    <div class="topbar">
      <img class="logo"
           src="<?= e($logo_url) ?>"
           <?= $logo_srcset ? 'srcset="'. $logo_srcset .'"' : '' ?>
           alt="Troya Imóveis"
           crossorigin="anonymous">
      <div class="right-top">FICHA DETALHADA DO IMÓVEL</div>
    </div>

    <div class="hero" <?php if ($hero): ?>
         style="background-image:url('<?= e($foto_url($hero)) ?>')"
         <?php endif; ?>>
      <?php if ($hero): ?>
        <div class="stripe">
          <div class="h-title"><?= e($title) ?></div>
          <div class="h-price"><?= e(fmt_money(v($imovel,'preco',null))) ?></div>
        </div>
      <?php endif; ?>
    </div>

    <!-- chips principais -->
    <div class="chips" style="margin-bottom:8pt;">
      <span class="chip"><?= e(($imovel['tipo'] ?? '—')).' • '.e(ucfirst($imovel['finalidade'] ?? '—')) ?></span>
      <?php if ($condominio_nome): ?><span class="chip">Condomínio: <?= e($condominio_nome) ?></span><?php endif; ?>
      <?php if ($bairro || $cidade || $uf): ?>
        <span class="chip"><?= e(trim(($bairro ? $bairro.' · ' : '').$cidade.( $uf ? '/'.$uf : '' ))) ?></span>
      <?php endif; ?>
    </div>

    <!-- 6 specs principais -->
    <div class="specs">
      <div class="kv"><div class="label">Preço</div><div class="value"><?= e(fmt_money(v($imovel,'preco',null))) ?></div></div>
      <div class="kv"><div class="label">Tipo</div><div class="value"><?= e(ucfirst(v($imovel,'tipo'))) ?></div></div>
      <div class="kv"><div class="label">Finalidade</div><div class="value"><?= e(ucfirst(v($imovel,'finalidade'))) ?></div></div>
      <div class="kv"><div class="label">Condomínio</div><div class="value"><?= e($condominio_nome ?: '—') ?></div></div>
      <div class="kv"><div class="label">Área construída (m²)</div><div class="value"><?= e(v($imovel,'area_construida_m2')) ?></div></div>
      <div class="kv"><div class="label">Área do lote (m²)</div><div class="value"><?= e(v($imovel,'area_lote_m2')) ?></div></div>
    </div>

    <!-- badges (só os que tiverem valor) -->
    <div class="badges">
      <?php if (v($imovel,'quartos','')!==''): ?><div class="badge">Quartos: <?= e(v($imovel,'quartos')) ?></div><?php endif; ?>
      <?php if (v($imovel,'suites','')!==''): ?><div class="badge">Suítes: <?= e(v($imovel,'suites')) ?></div><?php endif; ?>
      <?php if (v($imovel,'banheiros','')!==''): ?><div class="badge">Banheiros: <?= e(v($imovel,'banheiros')) ?></div><?php endif; ?>
      <?php if (v($imovel,'vagas','')!==''): ?><div class="badge">Vagas: <?= e(v($imovel,'vagas')) ?></div><?php endif; ?>
    </div>

    <!-- características (resumo) -->
    <?php if (!empty($caracts)): ?>
      <div class="section-title">Características do imóvel</div>
      <div class="chips">
        <?php
          $maxCap = 12; $i = 0;
          foreach ($caracts as $c) {
            if ($i++ >= $maxCap) break;
            echo '<span class="chip">'.e($c['nome']).'</span>';
          }
          if (count($caracts) > $maxCap) echo '<span class="chip">+ '.(count($caracts)-$maxCap).' mais</span>';
        ?>
      </div>
    <?php endif; ?>

    <!-- descrição curta (limitada visualmente) -->
    <?php if (!empty($imovel['descricao'])): ?>
      <div class="desc-title">Descrição</div>
      <div class="desc"><?= nl2br(e($imovel['descricao'])) ?></div>
    <?php endif; ?>

    <div class="footer">
      <div>Gerado em: <?= date('d/m/Y H:i') ?></div>
      <div>Troya Imóveis</div>
    </div>
  </div>
</div>

<?php
/* ====== FOTOS (2 por página, background cover) ====== */
for ($i = 0; $i < count($fotos); $i += 2):
  $f1 = $fotos[$i]   ?? null;
  $f2 = $fotos[$i+1] ?? null;
?>
<div class="sheet" data-sheet="foto">
  <div class="inner photos photo-page">
    <div class="frame" <?php if ($f1): ?>
         style="background-image:url('<?= e($foto_url($f1)) ?>')"
         <?php endif; ?>>
      <?php if ($f1): ?><span class="legend">Foto <?= ($i+1) ?></span><?php endif; ?>
    </div>
    <div class="frame" <?php if ($f2): ?>
         style="background-image:url('<?= e($foto_url($f2)) ?>')"
         <?php endif; ?>>
      <?php if ($f2): ?><span class="legend">Foto <?= ($i+2) ?></span><?php endif; ?>
    </div>
    <div class="footer">
      <div><?= e($title) ?></div>
      <div>Troya Imóveis</div>
    </div>
  </div>
</div>
<?php endfor; ?>

<div class="actions">
  <a class="btn secondary" href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/fotos">Voltar</a>
</div>

<!-- CDNs -->
<script src="https://cdn.jsdelivr.net/npm/html2canvas@1.4.1/dist/html2canvas.min.js" crossorigin="anonymous"></script>
<script src="https://cdn.jsdelivr.net/npm/jspdf@2.5.1/dist/jspdf.umd.min.js" crossorigin="anonymous"></script>
<script>
(function(){
  const { jsPDF } = window.jspdf;
  const bar = document.getElementById('bar');
  const sheets = Array.from(document.querySelectorAll('.sheet'));

  function updateBar(p){ bar.style.width = Math.max(1, Math.min(100, p)) + '%'; }

  function waitImages(){
    // como usamos background-image, não há <img> a esperar; ainda assim aguardamos um pequeno delay
    return new Promise(res=>setTimeout(res, 50));
  }

  async function buildPDF(){
    try{
      updateBar(5);
      await waitImages();

      // ↑ resolução mais alta sem mudar tamanho visual
      const DPR = Math.min(3, (window.devicePixelRatio || 1) * 1.5);

      const pdf = new jsPDF({ orientation:'p', unit:'pt', format:'a4', compress:true });
      const W = pdf.internal.pageSize.getWidth();
      const H = pdf.internal.pageSize.getHeight();

      for (let i=0; i<sheets.length; i++){
        const el = sheets[i];
        const canvas = await html2canvas(el, {
          backgroundColor:'#FFFFFF',
          scale: DPR,          // qualidade da captura
          useCORS: true,
          logging: false,
          imageTimeout: 15000
        });
        const imgData = canvas.toDataURL('image/jpeg', 0.95); // menos compressão (melhor qualidade)
        if (i>0) pdf.addPage();
        pdf.addImage(imgData, 'JPEG', 0, 0, W, H);
        updateBar(5 + (i+1) * (95 / sheets.length));
      }
      pdf.save('ficha-imovel-<?= (int)$imovel['id'] ?>.pdf');
      updateBar(100);
    }catch(err){
      console.error(err);
      alert('Não foi possível gerar o PDF no navegador. Tente novamente.');
    }
  }
  buildPDF();
})();
</script>
</body>
</html>