<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION["logged_in"]) || !$_SESSION["logged_in"]) {
    header("Location: /troyacrm/login");
    exit;
}

require_once __DIR__ . '/../../app/db.php';

/* === RBAC: exigir permissão para EDITAR imóveis === */
require_once __DIR__ . '/../../app/rbac.php';
rbac_require('imoveis','update');

$page_title  = 'Editar Imóvel';
$me_id       = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

// Verificar se ID foi passado
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: " . BASE_URL . "/imoveis");
    exit;
}

$property_id = intval($_GET['id']);

// Buscar dados do imóvel
$imovel = fetch_one("SELECT * FROM properties WHERE id = ?", [$property_id], true);
if (!$imovel) {
    header("Location: " . BASE_URL . "/imoveis");
    exit;
}

/* === RBAC: se o escopo de UPDATE for 'own', só permite editar se for dono === */
$scopeUpdate = rbac_scope('imoveis','update'); // 'none' | 'own' | 'all'
if ($scopeUpdate === 'own') {
    // Permite editar se owner_user_id estiver vazio OU for do usuário atual
    if (isset($imovel['owner_user_id']) && (int)$imovel['owner_user_id'] !== $me_id) {
        http_response_code(403);
        exit('Você não tem permissão para editar este imóvel.');
    }
}

/* === Tipos fixos (igual ao CRIAR) === */
$TIPOS_IMOVEL = [
    'Casa em Condomínio',
    'Sobrado em Condominio',
    'Casa de Bairro',
    'Apartamento'
];

/* === Carrega lista de condomínios (igual ao CRIAR) === */
$condominios = [];
try {
    $condominios = fetch_all("SELECT id, nome FROM condominios ORDER BY nome ASC", [], true) ?: [];
} catch (Throwable $e) {
    $condominios = [];
}

/* === Características selecionadas === */
$features_selecionadas = fetch_all(
    "SELECT feature_id FROM property_feature_values WHERE property_id = ?", 
    [$property_id], 
    true
);
$features_ids = array_column($features_selecionadas, 'feature_id');

$erro = null;
$sucesso = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $titulo = trim($_POST['titulo'] ?? '');
        $preco = floatval($_POST['preco'] ?? 0);
        $cidade = trim($_POST['cidade'] ?? '');
        $bairro = trim($_POST['bairro'] ?? '');
        $endereco = trim($_POST['endereco'] ?? '');
        $cep = trim($_POST['cep'] ?? '');
        $uf = trim($_POST['uf'] ?? '');
        $status = trim($_POST['status'] ?? 'rascunho');
        $finalidade = trim($_POST['finalidade'] ?? '');
        $tipo = trim($_POST['tipo'] ?? '');
        $condominio_id = (int)($_POST['condominio_id'] ?? 0);
        $quartos = intval($_POST['quartos'] ?? 0);
        $suites = intval($_POST['suites'] ?? 0);
        $banheiros = intval($_POST['banheiros'] ?? 0);
        $vagas = intval($_POST['vagas'] ?? 0);
        $area_construida = floatval($_POST['area_construida_m2'] ?? 0);
        $area_lote = floatval($_POST['area_lote_m2'] ?? 0);
        $descricao = trim($_POST['descricao'] ?? '');
        $features = $_POST['features'] ?? [];

        if (empty($titulo)) {
            throw new Exception('O título é obrigatório');
        }
        if (empty($endereco)) {
            throw new Exception('O endereço é obrigatório');
        }

        // Atualizar imóvel (inclui condominio_id)
        $sql = "UPDATE properties SET 
                titulo = ?, preco = ?, cidade = ?, bairro = ?, endereco = ?, cep = ?, uf = ?, 
                status = ?, finalidade = ?, tipo = ?, condominio_id = ?, quartos = ?, suites = ?, banheiros = ?, 
                vagas = ?, area_construida_m2 = ?, area_lote_m2 = ?, descricao = ?
                WHERE id = ?";

        execute($sql, [
            $titulo, $preco, $cidade, $bairro, $endereco, $cep, $uf, $status, $finalidade, $tipo,
            $condominio_id, $quartos, $suites, $banheiros, $vagas, $area_construida, $area_lote, $descricao,
            $property_id
        ]);
        
        // Registrar log de edição
        $detalhes = 'Edição do imóvel "' . $titulo . '" (ID ' . $property_id . ')';
        execute(
            "INSERT INTO property_logs (property_id, user_id, acao, detalhes) VALUES (?, ?, 'editado', ?)",
            [$property_id, $me_id, $detalhes]
        );
        
        // Atualizar características
        execute("DELETE FROM property_feature_values WHERE property_id = ?", [$property_id]);
        if (!empty($features) && is_array($features)) {
            foreach ($features as $feature_id) {
                if (is_numeric($feature_id)) {
                    execute(
                        "INSERT IGNORE INTO property_feature_values (property_id, feature_id) VALUES (?, ?)", 
                        [$property_id, intval($feature_id)]
                    );
                }
            }
        }
        
        $sucesso = true;
        
        // Recarregar dados atualizados
        $imovel = fetch_one("SELECT * FROM properties WHERE id = ?", [$property_id], true);
        $features_selecionadas = fetch_all(
            "SELECT feature_id FROM property_feature_values WHERE property_id = ?", 
            [$property_id], 
            true
        );
        $features_ids = array_column($features_selecionadas, 'feature_id');

    } catch (Exception $e) {
        $erro = $e->getMessage();
    }
}

// Buscar características disponíveis
$features_imovel = fetch_all("SELECT * FROM property_features WHERE categoria = 'imovel' ORDER BY nome ASC", [], true);
$features_condominio = fetch_all("SELECT * FROM property_features WHERE categoria = 'condominio' ORDER BY nome ASC", [], true);

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>

<style>
.dropdown-features {
    max-height: 400px;
    overflow-y: auto;
    padding: 15px;
    min-width: 350px;
}
.feature-group {
    margin-bottom: 15px;
}
.feature-group-title {
    font-weight: bold;
    color: #495057;
    margin-bottom: 10px;
    padding-bottom: 5px;
    border-bottom: 2px solid #dee2e6;
}
.feature-checkbox {
    margin-bottom: 8px;
}
.feature-checkbox label {
    cursor: pointer;
    user-select: none;
}
.feature-checkbox:hover {
    background-color: #f8f9fa;
    border-radius: 4px;
}
#selectedFeaturesCount {
    background-color: #198754;
    color: white;
    border-radius: 50%;
    padding: 2px 8px;
    font-size: 12px;
    font-weight: bold;
    margin-left: 5px;
}
</style>

<div class="container mt-4 mb-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="fw-bold mb-1">
                <i class="bi bi-pencil-square text-primary me-2"></i>
                Editar Imóvel
            </h1>
            <p class="text-muted mb-0"><?= htmlspecialchars($imovel['titulo']) ?></p>
        </div>
        <div class="d-flex gap-2">
            <a href="<?=BASE_URL?>/imoveis/<?=$property_id?>/fotos" class="btn btn-outline-primary">
                <i class="bi bi-images me-2"></i>Gerenciar Fotos
            </a>
            <a href="<?=BASE_URL?>/imoveis" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Voltar
            </a>
        </div>
    </div>

    <?php if ($sucesso): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <h5><i class="bi bi-check-circle me-2"></i>Sucesso!</h5>
            <p class="mb-0">Imóvel atualizado com sucesso!</p>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <?php if ($erro): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle me-2"></i><?=htmlspecialchars($erro)?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <form method="POST" class="card p-4 shadow-sm">
        <div class="row g-3">
            <!-- Título -->
            <div class="col-md-8">
                <label class="form-label fw-bold">Título *</label>
                <input type="text" name="titulo" class="form-control" value="<?=htmlspecialchars($imovel['titulo'])?>" required>
            </div>

            <!-- Preço -->
            <div class="col-md-4">
                <label class="form-label fw-bold">Preço (R$)</label>
                <input type="number" step="0.01" name="preco" class="form-control" value="<?=$imovel['preco']?>">
            </div>

            <div class="col-12"><hr></div>

            <!-- Finalidade -->
            <div class="col-md-4">
                <label class="form-label fw-bold">Finalidade</label>
                <select name="finalidade" class="form-select">
                    <option value="">Selecione</option>
                    <option value="venda" <?=$imovel['finalidade']=='venda'?'selected':''?>>Venda</option>
                    <option value="aluguel" <?=$imovel['finalidade']=='aluguel'?'selected':''?>>Aluguel</option>
                </select>
            </div>

            <!-- Tipo (SELECT igual ao CRIAR) -->
            <div class="col-md-4">
                <label class="form-label fw-bold">Tipo</label>
                <?php
                  $tipoAtual = (string)($imovel['tipo'] ?? '');
                  $inLista   = in_array($tipoAtual, $TIPOS_IMOVEL, true);
                ?>
                <select name="tipo" class="form-select">
                    <option value="">Selecione</option>
                    <?php if ($tipoAtual && !$inLista): ?>
                      <option value="<?= htmlspecialchars($tipoAtual) ?>" selected>[Atual] <?= htmlspecialchars($tipoAtual) ?></option>
                    <?php endif; ?>
                    <?php foreach ($TIPOS_IMOVEL as $t): ?>
                      <option value="<?= htmlspecialchars($t) ?>" <?= ($tipoAtual === $t ? 'selected' : '') ?>>
                        <?= htmlspecialchars($t) ?>
                      </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <!-- Status -->
            <div class="col-md-4">
                <label class="form-label fw-bold">Status</label>
                <select name="status" class="form-select">
                    <option value="publicado" <?=$imovel['status']=='publicado'?'selected':''?>>Publicado</option>
                    <option value="rascunho"  <?=$imovel['status']=='rascunho'?'selected':''?>>Rascunho</option>
                    <option value="vendido"   <?=$imovel['status']=='vendido'?'selected':''?>>Vendido</option>
                    <option value="alugado"   <?=$imovel['status']=='alugado'?'selected':''?>>Alugado</option>
                </select>
            </div>

            <div class="col-12"><hr></div>

            <!-- Condomínio (igual ao CRIAR) -->
            <div class="col-md-5">
                <label class="form-label fw-bold">Condomínio</label>
                <select name="condominio_id" class="form-select">
                    <option value="">Selecione</option>
                    <?php
                      $condSel = (int)($imovel['condominio_id'] ?? 0);
                      foreach ($condominios as $c):
                        $idc = (int)$c['id'];
                        $nm  = (string)$c['nome'];
                    ?>
                      <option value="<?= $idc ?>" <?= ($condSel === $idc ? 'selected' : '') ?>>
                        <?= htmlspecialchars($nm) ?>
                      </option>
                    <?php endforeach; ?>
                </select>
                <small class="text-muted">
                    Precisa cadastrar um condomínio? Acesse <a href="<?= BASE_URL ?>/admin/condominios" target="_blank">Admin → Condomínios</a>.
                </small>
            </div>

            <!-- CEP -->
            <div class="col-md-3">
                <label class="form-label fw-bold">CEP</label>
                <input type="text" name="cep" id="cep" class="form-control" value="<?=htmlspecialchars($imovel['cep']??'')?>" maxlength="9">
            </div>

            <!-- UF -->
            <div class="col-md-3">
                <label class="form-label fw-bold">UF</label>
                <input type="text" name="uf" id="uf" maxlength="2" class="form-control text-uppercase" value="<?=htmlspecialchars($imovel['uf']??'')?>">
            </div>

            <!-- Endereço -->
            <div class="col-md-9">
                <label class="form-label fw-bold">Endereço *</label>
                <input type="text" name="endereco" id="endereco" class="form-control" value="<?=htmlspecialchars($imovel['endereco'])?>" required>
            </div>

            <!-- Bairro -->
            <div class="col-md-3">
                <label class="form-label fw-bold">Bairro</label>
                <input type="text" name="bairro" id="bairro" class="form-control" value="<?=htmlspecialchars($imovel['bairro']?? '')?>">
            </div>

            <!-- Cidade -->
            <div class="col-md-6">
                <label class="form-label fw-bold">Cidade</label>
                <input type="text" name="cidade" id="cidade" class="form-control" value="<?=htmlspecialchars($imovel['cidade'])?>">
            </div>

            <div class="col-12"><hr></div>

            <!-- Quartos -->
            <div class="col-md-2">
                <label class="form-label fw-bold">Quartos</label>
                <input type="number" name="quartos" class="form-control" min="0" value="<?=$imovel['quartos']??0?>">
            </div>

            <!-- Suítes -->
            <div class="col-md-2">
                <label class="form-label fw-bold">Suítes</label>
                <input type="number" name="suites" class="form-control" min="0" value="<?=$imovel['suites']??0?>">
            </div>

            <!-- Banheiros -->
            <div class="col-md-2">
                <label class="form-label fw-bold">Banheiros</label>
                <input type="number" name="banheiros" class="form-control" min="0" value="<?=$imovel['banheiros']??0?>">
            </div>

            <!-- Vagas -->
            <div class="col-md-2">
                <label class="form-label fw-bold">Vagas</label>
                <input type="number" name="vagas" class="form-control" min="0" value="<?=$imovel['vagas']??0?>">
            </div>

            <!-- Área Construída -->
            <div class="col-md-2">
                <label class="form-label fw-bold">Área Const. (m²)</label>
                <input type="number" step="0.01" name="area_construida_m2" class="form-control" value="<?=$imovel['area_construida_m2']??''?>">
            </div>

            <!-- Área do Lote -->
            <div class="col-md-2">
                <label class="form-label fw-bold">Área Lote (m²)</label>
                <input type="number" step="0.01" name="area_lote_m2" class="form-control" value="<?=$imovel['area_lote_m2']??''?>">
            </div>

            <div class="col-12"><hr></div>

            <!-- Características -->
            <div class="col-12">
                <label class="form-label fw-bold">Características do Imóvel</label>
                <div class="dropdown">
                    <button class="btn btn-outline-secondary dropdown-toggle w-100 text-start" type="button" id="dropdownFeatures" data-bs-toggle="dropdown" data-bs-auto-close="outside">
                        <i class="bi bi-check2-square me-2"></i>Selecionar características
                        <span id="selectedFeaturesCount" style="<?=count($features_ids)>0?'':'display:none'?>"><?=count($features_ids)?></span>
                    </button>
                    <div class="dropdown-menu dropdown-features">
                        <!-- Características do Imóvel -->
                        <div class="feature-group">
                            <div class="feature-group-title">
                                <i class="bi bi-house-door me-2"></i>Características do Imóvel
                            </div>
                            <?php foreach ($features_imovel as $feature): ?>
                                <div class="form-check feature-checkbox">
                                    <input class="form-check-input feature-check" type="checkbox" name="features[]" 
                                           value="<?=$feature['id']?>" id="feat<?=$feature['id']?>"
                                           <?=in_array($feature['id'], $features_ids)?'checked':''?>>
                                    <label class="form-check-label" for="feat<?=$feature['id']?>">
                                        <i class="<?=$feature['icone']?> me-1"></i>
                                        <?=$feature['nome']?>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>

                        <!-- Características do Condomínio -->
                        <div class="feature-group">
                            <div class="feature-group-title">
                                <i class="bi bi-building me-2"></i>Características do Condomínio
                            </div>
                            <?php foreach ($features_condominio as $feature): ?>
                                <div class="form-check feature-checkbox">
                                    <input class="form-check-input feature-check" type="checkbox" name="features[]" 
                                           value="<?=$feature['id']?>" id="feat<?=$feature['id']?>"
                                           <?=in_array($feature['id'], $features_ids)?'checked':''?>>
                                    <label class="form-check-label" for="feat<?=$feature['id']?>">
                                        <i class="<?=$feature['icone']?> me-1"></i>
                                        <?=$feature['nome']?>
                                    </label>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-12"><hr></div>

            <!-- Descrição -->
            <div class="col-12">
                <label class="form-label fw-bold">Descrição</label>
                <textarea name="descricao" rows="4" class="form-control"><?=htmlspecialchars($imovel['descricao']??'')?></textarea>
            </div>

            <!-- Botões -->
            <div class="col-12 mt-4 d-flex gap-2">
                <button type="submit" class="btn btn-primary btn-lg">
                    <i class="bi bi-check-circle me-2"></i>Salvar Alterações
                </button>
                <a href="<?=BASE_URL?>/imoveis" class="btn btn-outline-secondary btn-lg">
                    <i class="bi bi-x-circle me-2"></i>Cancelar
                </a>
            </div>
        </div>
    </form>
</div>

<script>
// Buscar CEP
document.getElementById('cep').addEventListener('blur', function() {
    const cep = this.value.replace(/\D/g, '');
    if (cep.length === 8) {
        fetch(`https://viacep.com.br/ws/${cep}/json/`)
            .then(res => res.json())
            .then(data => {
                if (!data.erro) {
                    document.getElementById('endereco').value = data.logradouro || '';
                    document.getElementById('bairro').value = data.bairro || '';
                    document.getElementById('cidade').value = data.localidade || '';
                    document.getElementById('uf').value = data.uf || '';
                    document.getElementById('endereco').focus();
                }
            });
    }
});

// Contador de características
document.querySelectorAll('.feature-check').forEach(checkbox => {
    checkbox.addEventListener('change', updateFeatureCount);
});

function updateFeatureCount() {
    const count = document.querySelectorAll('.feature-check:checked').length;
    const badge = document.getElementById('selectedFeaturesCount');
    if (count > 0) {
        badge.textContent = count;
        badge.style.display = 'inline-block';
    } else {
        badge.style.display = 'none';
    }
}

// Máscara CEP
document.getElementById('cep').addEventListener('input', function(e) {
    let value = e.target.value.replace(/\D/g, '');
    if (value.length > 5) {
        value = value.substring(0, 5) + '-' + value.substring(5, 8);
    }
    e.target.value = value;
});
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>