<?php
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION["logged_in"])) { header("Location: /troyacrm/login"); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

/* ==== Helpers simples ==== */
function hv($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function pick(array $row, array $keys){
  foreach ($keys as $k) { if (isset($row[$k]) && $row[$k] !== '' && $row[$k] !== null) return $row[$k]; }
  return null;
}

/* ==== RBAC: precisa ver clientes ==== */
rbac_require('clientes','view');

/* ==== ID ==== */
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) { header("Location: " . BASE_URL . "/clientes"); exit; }

/* ==== Carrega cliente ==== */
try {
  $cliente = fetch_one("SELECT * FROM clients WHERE id = ?", [$id], true);
  if (!$cliente) { header("Location: " . BASE_URL . "/clientes"); exit; }

  // Se o escopo de view for 'own', só permite ver se for dono/criador
  $scopeView = rbac_scope('clientes','view'); // none|own|all
  if ($scopeView === 'own') {
    $me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
    $owner = isset($cliente['owner_user_id']) ? (int)$cliente['owner_user_id'] : (int)($cliente['created_by'] ?? 0);
    if ($owner !== $me_id) { http_response_code(403); exit('Você não tem permissão para visualizar este cliente.'); }
  }
} catch (Throwable $e) {
  die("Erro ao carregar cliente: " . $e->getMessage());
}

/* ==== Dono/Responsável ==== */
$owner_user_id = (int)($cliente['owner_user_id'] ?? $cliente['created_by'] ?? 0);
$owner_display = $owner_user_id ? ('Usuário #'.$owner_user_id) : 'Não definido';
$owner_email   = null;
if ($owner_user_id) {
  try {
    $u = fetch_one("SELECT id, nome, email FROM users WHERE id = ?", [$owner_user_id], true);
    if ($u) {
      $owner_display = !empty($u['nome']) ? $u['nome'] : (!empty($u['email']) ? $u['email'] : $owner_display);
      $owner_email   = $u['email'] ?? null;
    }
  } catch (Throwable $e) { /* ok */ }
}

/* ==== Campos extras com tolerância de nomes ==== */
$canal_captacao   = pick($cliente, ['canal_captacao','canal','origem','fonte_captacao','canal_origem','origem_cadastro']);
$como_conheceu    = pick($cliente, ['como_conheceu','indicacao_de','referencia','como_nos_conheceu']);
$status_cli       = pick($cliente, ['status','situacao']);

$estado_civil     = pick($cliente, ['estado_civil','estadoCivil']);
$profissao        = pick($cliente, ['profissao','ocupacao']);
$renda            = pick($cliente, ['renda','renda_mensal','renda_estimada']);

$tipo_interesse   = pick($cliente, ['tipo_interesse','tipo_imovel_interesse']);
$faixa_preco      = pick($cliente, ['faixa_preco','orcamento','orcamento_max','orcamento_min_max']);
$regiao_interesse = pick($cliente, ['regiao_interesse','bairros_interesse','bairro_interesse']);
$cidade_interesse = pick($cliente, ['cidade_interesse','cidade_desejada']);

/* ==== Título ==== */
$page_title = "Cliente: " . hv($cliente['nome_completo']);

/* ==== Pode editar este registro? ==== */
$can_edit_this = false;
if (rbac_can('clientes','update')) {
  $scopeUpd = rbac_scope('clientes','update'); // none|own|all
  if ($scopeUpd === 'all') {
    $can_edit_this = true;
  } elseif ($scopeUpd === 'own') {
    $me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
    $owner = isset($cliente['owner_user_id']) ? (int)$cliente['owner_user_id'] : (int)($cliente['created_by'] ?? 0);
    $can_edit_this = ($owner === $me_id);
  }
}

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<style>
/* Header (mantém seu visual) */
.client-header {
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  color: white; padding: 2rem; border-radius: 15px; margin-bottom: 2rem;
  box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}
.info-card {
  background: var(--bs-dark); border: 1px solid rgba(255,255,255,0.1);
  border-radius: 10px; padding: 1.5rem;
}
.info-label { color: #adb5bd; font-size: .875rem; font-weight: 500; margin-bottom:.25rem; text-transform: uppercase; letter-spacing:.5px; }
.info-value { color: #e9ecef; font-size: 1.06rem; font-weight: 500; margin-bottom: 0; }
.info-value a { color: #0dcaf0; text-decoration: none; }
.info-value a:hover { color: #0aa4c0; text-decoration: underline; }

/* Masonry: evita buracos entre cards */
.masonry { column-count: 1; column-gap: 1rem; }
@media (min-width: 992px) { .masonry { column-count: 2; } }
.masonry .info-card {
  display: inline-block; width: 100%; break-inside: avoid; -webkit-column-break-inside: avoid; margin: 0 0 1rem;
}

/* Toolbar flutuante */
.toolbar-fixed {
  position: fixed; bottom: 20px; right: 20px; z-index: 1050;
  box-shadow: 0 4px 12px rgba(0,0,0,0.3);
}
@media (max-width: 768px) {
  .client-header { padding: 1.5rem; margin-bottom: 1rem; }
  .info-card { padding: 1rem; }
  .toolbar-fixed { bottom: 10px; right: 10px; left: 10px; }
  .toolbar-fixed .btn-group { width: 100%; flex-wrap: wrap; }
}
.badge-pill {
  background: rgba(255,255,255,.15); color:#fff; padding:.35rem .65rem; border-radius:999px; font-weight:600;
}
</style>

<div class="container mt-4 mb-5">
  <!-- HEADER -->
  <div class="client-header">
    <div class="row align-items-center">
      <div class="col-md-8">
        <h1 class="mb-1"><?= hv($cliente['nome_completo']) ?></h1>
        <div class="d-flex flex-wrap gap-2 mb-2">
          <span class="badge-pill"><i class="bi bi-hash me-1"></i>#<?= (int)$cliente['id'] ?></span>

          <?php
          $tipos = ['comprador'=>'Comprador','vendedor'=>'Vendedor','locatario'=>'Locatário','proprietario'=>'Proprietário'];
          if (!empty($cliente['tipo_cliente']) && isset($tipos[$cliente['tipo_cliente']])): ?>
            <span class="badge-pill"><?= hv($tipos[$cliente['tipo_cliente']]) ?></span>
          <?php endif; ?>

          <?php if (!empty($cliente['created_at'])): ?>
            <span class="badge-pill"><i class="bi bi-calendar me-1"></i><?= date('d/m/Y', strtotime($cliente['created_at'])) ?></span>
          <?php endif; ?>

          <?php if ($owner_user_id): ?>
            <span class="badge-pill"><i class="bi bi-person-check me-1"></i>Resp.: <?= hv($owner_display) ?></span>
          <?php endif; ?>

          <?php if ($canal_captacao): ?>
            <span class="badge-pill"><i class="bi bi-megaphone me-1"></i><?= hv($canal_captacao) ?></span>
          <?php endif; ?>
        </div>
      </div>

      <div class="col-md-4 text-md-end">
        <?php if (!empty($cliente['email'])): ?>
          <a href="mailto:<?= hv($cliente['email']) ?>" class="btn btn-outline-light btn-sm me-2">
            <i class="bi bi-envelope me-1"></i>E-mail
          </a>
        <?php endif; ?>
        <?php if (!empty($cliente['telefone_principal'])): ?>
          <a href="https://wa.me/55<?= preg_replace('/\D/','',$cliente['telefone_principal']) ?>" target="_blank" class="btn btn-success btn-sm">
            <i class="bi bi-whatsapp me-1"></i>WhatsApp
          </a>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <?php
    $hasResumo = (bool)($canal_captacao || $como_conheceu || $status_cli);
    $hasPrefs  = (bool)($tipo_interesse || $faixa_preco || $regiao_interesse || $cidade_interesse);
  ?>

  <!-- CARDS EM MASONRY (sem lacunas) -->
  <div class="masonry">

    <?php if ($hasResumo): ?>
    <section class="info-card">
      <h5 class="mb-3 text-info"><i class="bi bi-clipboard-data me-2"></i>Resumo do Lead</h5>

      <?php if ($canal_captacao): ?>
        <div class="mb-2">
          <div class="info-label">Canal de captação</div>
          <p class="info-value"><?= hv($canal_captacao) ?></p>
        </div>
      <?php endif; ?>

      <?php if ($como_conheceu): ?>
        <div class="mb-2">
          <div class="info-label">Como nos conheceu</div>
          <p class="info-value"><?= hv($como_conheceu) ?></p>
        </div>
      <?php endif; ?>

      <?php if ($status_cli): ?>
        <div class="mb-0">
          <div class="info-label">Status</div>
          <p class="info-value"><?= hv($status_cli) ?></p>
        </div>
      <?php endif; ?>
    </section>
    <?php endif; ?>

    <section class="info-card">
      <h5 class="mb-3 text-success"><i class="bi bi-telephone me-2"></i>Contato</h5>

      <?php if (!empty($cliente['telefone_principal'])): ?>
        <div class="mb-2">
          <div class="info-label">Telefone principal</div>
          <p class="info-value">
            <i class="bi bi-telephone me-1"></i>
            <a href="tel:<?= hv($cliente['telefone_principal']) ?>"><?= hv($cliente['telefone_principal']) ?></a>
          </p>
        </div>
      <?php endif; ?>

      <?php if (!empty($cliente['telefone_secundario'])): ?>
        <div class="mb-2">
          <div class="info-label">Telefone secundário</div>
          <p class="info-value">
            <a href="tel:<?= hv($cliente['telefone_secundario']) ?>"><?= hv($cliente['telefone_secundario']) ?></a>
          </p>
        </div>
      <?php endif; ?>

      <?php if (!empty($cliente['email'])): ?>
        <div class="mb-2">
          <div class="info-label">E-mail</div>
          <p class="info-value">
            <a href="mailto:<?= hv($cliente['email']) ?>"><?= hv($cliente['email']) ?></a>
          </p>
        </div>
      <?php endif; ?>

      <?php if (!empty($cliente['created_at'])): ?>
        <hr class="text-secondary">
        <div class="info-label">Cadastrado em</div>
        <p class="info-value"><?= date('d/m/Y H:i', strtotime($cliente['created_at'])) ?></p>
      <?php endif; ?>

      <?php if ($owner_user_id): ?>
        <div class="info-label mt-2">Cadastrado por</div>
        <p class="info-value">
          <?php if ($owner_email): ?>
            <a href="mailto:<?= hv($owner_email) ?>"><?= hv($owner_display) ?></a>
          <?php else: ?>
            <?= hv($owner_display) ?>
          <?php endif; ?>
        </p>
      <?php endif; ?>

      <?php if (!empty($cliente['updated_at']) && $cliente['updated_at'] != $cliente['created_at']): ?>
        <div class="info-label mt-2">Última atualização</div>
        <p class="info-value text-muted"><?= date('d/m/Y H:i', strtotime($cliente['updated_at'])) ?></p>
      <?php endif; ?>
    </section>

    <section class="info-card">
      <h5 class="mb-3 text-primary"><i class="bi bi-person-circle me-2"></i>Informações Pessoais</h5>

      <?php if (!empty($cliente['cpf_cnpj'])): ?>
        <div class="mb-2">
          <div class="info-label">CPF/CNPJ</div>
          <p class="info-value"><?= hv($cliente['cpf_cnpj']) ?></p>
        </div>
      <?php endif; ?>

      <?php if (!empty($cliente['data_nascimento'])): ?>
        <div class="mb-2">
          <div class="info-label">Data de nascimento</div>
          <p class="info-value"><?= date('d/m/Y', strtotime($cliente['data_nascimento'])) ?></p>
        </div>
      <?php endif; ?>

      <?php if ($estado_civil): ?>
        <div class="mb-2">
          <div class="info-label">Estado civil</div>
          <p class="info-value"><?= hv($estado_civil) ?></p>
        </div>
      <?php endif; ?>

      <?php if ($profissao): ?>
        <div class="mb-2">
          <div class="info-label">Profissão</div>
          <p class="info-value"><?= hv($profissao) ?></p>
        </div>
      <?php endif; ?>

      <?php if ($renda): ?>
        <hr class="text-secondary">
        <div class="info-label">Renda</div>
        <p class="info-value"><?= hv($renda) ?></p>
      <?php endif; ?>
    </section>

    <section class="info-card">
      <h5 class="mb-3 text-info"><i class="bi bi-geo-alt me-2"></i>Endereço</h5>

      <?php if (!empty($cliente['endereco'])): ?>
        <div class="mb-2">
          <div class="info-label">Endereço</div>
          <p class="info-value"><?= hv($cliente['endereco']) ?></p>
        </div>
      <?php endif; ?>

      <div class="row g-2">
        <?php if (!empty($cliente['bairro'])): ?>
          <div class="col-12 col-md-6">
            <div class="info-label">Bairro</div>
            <p class="info-value mb-2"><?= hv($cliente['bairro']) ?></p>
          </div>
        <?php endif; ?>
        <?php if (!empty($cliente['cidade'])): ?>
          <div class="col-8 col-md-4">
            <div class="info-label">Cidade</div>
            <p class="info-value mb-2"><?= hv($cliente['cidade']) ?></p>
          </div>
        <?php endif; ?>
        <?php if (!empty($cliente['uf'])): ?>
          <div class="col-4 col-md-2">
            <div class="info-label">UF</div>
            <p class="info-value mb-2"><?= hv($cliente['uf']) ?></p>
          </div>
        <?php endif; ?>
      </div>

      <?php if (!empty($cliente['cep'])): ?>
        <div class="mb-0">
          <div class="info-label">CEP</div>
          <p class="info-value"><?= hv($cliente['cep']) ?></p>
        </div>
      <?php endif; ?>
    </section>

    <?php if ($hasPrefs): ?>
    <section class="info-card">
      <h5 class="mb-3 text-warning"><i class="bi bi-stars me-2"></i>Preferências / Interesses</h5>
      <div class="row">
        <?php if ($tipo_interesse): ?>
          <div class="col-md-3 mb-2">
            <div class="info-label">Tipo de imóvel</div>
            <p class="info-value"><?= hv($tipo_interesse) ?></p>
          </div>
        <?php endif; ?>
        <?php if ($faixa_preco): ?>
          <div class="col-md-3 mb-2">
            <div class="info-label">Faixa de preço / orçamento</div>
            <p class="info-value"><?= hv($faixa_preco) ?></p>
          </div>
        <?php endif; ?>
        <?php if ($regiao_interesse): ?>
          <div class="col-md-3 mb-2">
            <div class="info-label">Região/Bairro de interesse</div>
            <p class="info-value"><?= hv($regiao_interesse) ?></p>
          </div>
        <?php endif; ?>
        <?php if ($cidade_interesse): ?>
          <div class="col-md-3 mb-2">
            <div class="info-label">Cidade de interesse</div>
            <p class="info-value"><?= hv($cidade_interesse) ?></p>
          </div>
        <?php endif; ?>
      </div>
    </section>
    <?php endif; ?>

    <?php if (!empty($cliente['observacoes'])): ?>
    <section class="info-card">
      <h5 class="mb-3 text-secondary"><i class="bi bi-chat-text me-2"></i>Observações</h5>
      <div class="info-value" style="white-space: pre-wrap; line-height: 1.55">
        <?= nl2br(hv($cliente['observacoes'])) ?>
      </div>
    </section>
    <?php endif; ?>

  </div><!-- /masonry -->
</div>

<!-- Toolbar fixa -->
<div class="toolbar-fixed">
  <div class="btn-group shadow-lg" role="group">
    <?php if ($can_edit_this): ?>
      <a href="<?= BASE_URL ?>/clientes/<?= (int)$cliente['id'] ?>/editar" class="btn btn-primary">
        <i class="bi bi-pencil me-2"></i>Editar
      </a>
    <?php endif; ?>
    <a href="<?= BASE_URL ?>/clientes" class="btn btn-secondary">
      <i class="bi bi-arrow-left me-2"></i>Voltar
    </a>
    <button class="btn btn-outline-light" onclick="window.print()">
      <i class="bi bi-printer me-2"></i>Imprimir
    </button>
  </div>
</div>

<script>
window.addEventListener('beforeprint', function(){ var tb=document.querySelector('.toolbar-fixed'); if(tb) tb.style.display='none'; });
window.addEventListener('afterprint',  function(){ var tb=document.querySelector('.toolbar-fixed'); if(tb) tb.style.display='block'; });
</script>

<?php include __DIR__ . '/../../layout/footer.php'; ?>