<?php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../../app/db.php';
require_once __DIR__ . '/../../../app/rbac.php';
require_once __DIR__ . '/../../../app/mailer.php'; // precisa do mailer_send_many()

$BASE  = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* Permissões básicas */
if (!rbac_can('atendimentos','view')) { http_response_code(403); exit('Sem permissão.'); }
$canUpd = rbac_can('atendimentos','update');
$scUpd  = rbac_scope('atendimentos','update'); // own|all|none

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* ===== helper local p/ redirecionar de volta ===== */
function back($ok=null,$err=null){
  global $BASE;
  $q = [];
  if ($ok)  $q['ok']  = $ok;
  if ($err) $q['err'] = $err;
  $qs = $q ? ('?'.http_build_query($q)) : '';
  header("Location: {$BASE}/admin/alertas{$qs}");
  exit;
}

/* ===== template local do e-mail (manual) — igual ao do CRON ===== */
if (!function_exists('alerts_build_inactivity_email_manual')) {
  /**
   * @param array $data [id, cliente_nome, status, last_dt]
   * @param string $base BASE_URL
   * @return array{subject:string, html:string}
   */
  function alerts_build_inactivity_email_manual(array $data, string $base): array {
    $id       = (int)($data['id'] ?? 0);
    $cliente  = (string)($data['cliente_nome'] ?? '—');
    $status   = (string)($data['status'] ?? '—');
    $lastRef  = $data['last_dt'] ?? null;

    $gapDays = 0;
    if ($lastRef) {
      try {
        $diff    = (new DateTime())->diff(new DateTime($lastRef));
        $gapDays = (int)$diff->format('%a');
      } catch (Throwable $e) { $gapDays = 0; }
    }

    $subject = "Alerta de inatividade • Atendimento #{$id} — {$cliente}";
    $link    = rtrim($base, '/') . '/atendimentos/visualizar?id=' . $id;

    $html = "
      <div style='font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;font-size:14px;color:#111'>
        <h2 style='margin:0 0 .6rem'>Alerta de inatividade</h2>

        <p style='margin:.4rem 0'>Atendimento <strong>#{$id}</strong> do cliente
          <strong>".htmlspecialchars($cliente)."</strong>.</p>

        <p style='margin:.4rem 0'>
          Última atividade: <strong>".($lastRef ? date('d/m/Y H:i', strtotime($lastRef)) : '—')."</strong>
          (≈ <strong>{$gapDays} dia(s)</strong> sem atualização).
        </p>

        <table cellpadding='6' cellspacing='0' style='border:1px solid #e5e7eb;border-radius:8px;margin:.6rem 0'>
          <tr>
            <td><strong>Status</strong></td>
            <td>".htmlspecialchars(ucfirst(str_replace('_',' ', $status)))."</td>
          </tr>
          <tr>
            <td><strong>Cliente</strong></td>
            <td>".htmlspecialchars($cliente)."</td>
          </tr>
          <tr>
            <td><strong>ID</strong></td>
            <td>#{$id}</td>
          </tr>
        </table>

        <p style='margin:.8rem 0 .2rem; font-weight:600'>Por favor, atualize este atendimento para não perder o cliente:</p>
        <ul style='margin:.2rem 0 .8rem .95rem; padding:0'>
          <li>Registre uma nova interação (ligação, WhatsApp, e-mail ou visita);</li>
          <li>Agende a próxima ação com data/hora;</li>
          <li>Se necessário, reatribua para outro responsável.</li>
        </ul>

        <p style='margin:1rem 0'>
          <a href='{$link}' style='background:#198754;color:#fff;padding:.55rem .9rem;border-radius:.45rem;text-decoration:none;display:inline-block'>
            Abrir atendimento
          </a>
        </p>

        <hr style='border:none;border-top:1px solid #ddd;margin:1rem 0'>
        <p style='color:#555'>Mensagem enviada manualmente pelo Troya CRM.</p>
      </div>
    ";

    return ['subject' => $subject, 'html' => $html];
  }
}

try {
  if ($_SERVER['REQUEST_METHOD'] !== 'POST') { back(null, 'Método inválido.'); }
  if (!hash_equals($CSRF, (string)($_POST['csrf'] ?? ''))) { back(null, 'CSRF inválido.'); }

  $action = (string)($_POST['action'] ?? '');
  $id     = (int)($_POST['id'] ?? 0);
  if ($id <= 0 || $action === '') { back(null, 'Requisição incompleta.'); }

  // Carrega o atendimento alvo + dono
  $row = fetch_one("
    SELECT i.id, i.owner_user_id, i.created_by,
           u.nome  AS owner_nome,
           u.email AS owner_email
      FROM interactions i
      LEFT JOIN users u ON u.id = i.owner_user_id
     WHERE i.id = ?
     LIMIT 1
  ", [$id], true);

  if (!$row) { back(null, 'Atendimento não encontrado.'); }

  // Helper de permissão por linha (para escopo own)
  $isOwn   = in_array($me_id, [(int)($row['owner_user_id'] ?? 0), (int)($row['created_by'] ?? 0)], true);
  $mustOwn = ($scUpd === 'own');

  switch ($action) {

    /* ============================
       ENVIO MANUAL DE E-MAIL (novo)
       ============================ */
    case 'notificar_email':
      if (!$canUpd)               back(null, 'Sem permissão para notificar.');
      if ($mustOwn && !$isOwn)    back(null, 'Sem permissão neste registro.');

      $ownerEmail = trim((string)($row['owner_email'] ?? ''));
      $ownerNome  = trim((string)($row['owner_nome']  ?? 'Corretor'));

      if ($ownerEmail === '') {
        back(null, 'O responsável por este atendimento não possui e-mail cadastrado.');
      }

      // Busca dados extras p/ montar assunto/corpo
      $det = fetch_one("
        SELECT i.id, i.status,
               COALESCE(
                 (SELECT MAX(l.created_at) FROM interaction_logs l WHERE l.interaction_id = i.id),
                 i.last_activity_at, i.updated_at, i.created_at
               ) AS last_dt,
               c.nome_completo AS cliente_nome
          FROM interactions i
          LEFT JOIN clients c ON c.id = i.client_id
         WHERE i.id = ?
         LIMIT 1
      ", [$id], true) ?: [];

      $tpl = alerts_build_inactivity_email_manual($det, $BASE);

      $enviados = mailer_send_many([['email'=>$ownerEmail, 'nome'=>$ownerNome]], $tpl['subject'], $tpl['html']);

      if ($enviados > 0) {
        try {
          execute("INSERT INTO system_logs (tipo, ref_id, mensagem, created_at) VALUES ('alerta_inatividade_manual', ?, ?, NOW())",
            [$row['id'], "E-mail manual enviado para 1 destinatário ({$ownerEmail})"]);
        } catch (Throwable $e) { /* tabela opcional */ }
        back('Alerta enviado para o corretor por e-mail.');
      } else {
        back(null, 'Falha ao enviar o e-mail ao corretor.');
      }
      break;

    /* ============================
       AÇÃO LEGADA: ADIAR (+X dias)
       ============================ */
    case 'notificar':
      if (!$canUpd)            back(null, 'Sem permissão para notificar.');
      if ($mustOwn && !$isOwn) back(null, 'Sem permissão neste registro.');

      $days = max(1, (int)($_POST['next_days'] ?? 2));
      execute(
        "UPDATE interactions SET next_followup_at = (NOW() + INTERVAL ? DAY), updated_at = NOW() WHERE id = ?",
        [$days, $id]
      );
      back('Corretor notificado (+'.$days.' dia(s)).');
      break;

    /* ============================
       ASSUMIR
       ============================ */
    case 'assumir':
      if (!$canUpd) { back(null, 'Sem permissão para assumir.'); }
      execute(
        "UPDATE interactions SET owner_user_id = ?, updated_at = NOW() WHERE id = ?",
        [$me_id, $id]
      );
      back('Atendimento assumido por você.');
      break;

    /* ============================
       REATRIBUIR
       ============================ */
    case 'reatribuir':
      $isAdminOrAll = rbac_can('admin','access') || ($scUpd === 'all');
      if (!$canUpd || !$isAdminOrAll) back(null, 'Sem permissão para reatribuir.');

      $to_user_id = (int)($_POST['to_user_id'] ?? 0);
      if ($to_user_id <= 0) back(null, 'Selecione o novo responsável.');

      $dst = fetch_one("SELECT id FROM users WHERE id = ?", [$to_user_id], true);
      if (!$dst) back(null, 'Responsável inválido.');

      execute(
        "UPDATE interactions SET owner_user_id = ?, updated_at = NOW() WHERE id = ?",
        [$to_user_id, $id]
      );
      back('Responsável reatribuído.');
      break;

    /* ============================
       SONECA
       ============================ */
    case 'soneca':
      if (!$canUpd)            back(null, 'Sem permissão para silenciar.');
      if ($mustOwn && !$isOwn) back(null, 'Sem permissão neste registro.');

      $snooze = max(1, (int)($_POST['snooze_days'] ?? 3));
      execute(
        "UPDATE interactions SET alerts_snooze_until = (NOW() + INTERVAL ? DAY), updated_at = NOW() WHERE id = ?",
        [$snooze, $id]
      );
      back('Alerta silenciado por '.$snooze.' dia(s).');
      break;

    default:
      back(null, 'Ação desconhecida.');
  }

} catch (Throwable $e) {
  back(null, $e->getMessage());
}