<?php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../../app/db.php';
require_once __DIR__ . '/../../../app/rbac.php';

$BASE       = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$page_title = 'Alertas de Inatividade';
$me_id      = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);

/* Permissões */
if (!rbac_can('atendimentos','view')) { http_response_code(403); exit('Sem permissão.'); }
$scopeV = rbac_scope('atendimentos','view'); // own|all
$canUpd = rbac_can('atendimentos','update');
$scUpd  = rbac_scope('atendimentos','update'); // own|all|none

/* CSRF */
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

/* Config (dias) */
$days = 3;
try {
  // Lê de "settings" (configurações)
  $cfg = fetch_one("SELECT value FROM settings WHERE `key`='alerts.days_without_update' LIMIT 1", [], true);
  if ($cfg && is_numeric($cfg['value'])) { $days = max(1, (int)$cfg['value']); }
} catch (Throwable $e) {
  // fallback em caso de ambientes antigos: system_configs
  try {
    $cfg = fetch_one("SELECT `value` FROM system_configs WHERE `key`='alerts_inactivity_days' LIMIT 1", [], true);
    if ($cfg && is_numeric($cfg['value'])) { $days = max(1, (int)$cfg['value']); }
  } catch (Throwable $e2) {}
}

$labelDias = ($days === 1) ? 'dia' : 'dias';

/* WHERE por dono + filtro de soneca */
$whereOwner = '';
$params = [$days];

if ($scopeV === 'own') {
  $whereOwner = " AND (i.owner_user_id = ? OR (i.owner_user_id IS NULL AND i.created_by = ?)) ";
  $params[] = $me_id;
  $params[] = $me_id;
}

/* Lista de usuários para reatribuir (se aplicável) */
$owners = [];
if ($canUpd && ($scUpd === 'all' || rbac_can('admin','access'))) {
  try { $owners = fetch_all("SELECT id, nome FROM users ORDER BY nome ASC", [], true) ?: []; } catch (Throwable $e) {}
}

/* Busca alertas (respeita soneca) */
$rows = [];
try {
  $rows = fetch_all("
    SELECT 
      i.id,
      i.client_id,
      i.status,
      i.next_followup_at,
      i.alerts_snooze_until,
      i.owner_user_id,
      i.created_by,
      c.nome_completo AS cliente_nome,
      u.nome          AS owner_nome,
      COALESCE(
        (SELECT MAX(l.created_at) FROM interaction_logs l WHERE l.interaction_id = i.id),
        i.last_activity_at,
        i.updated_at,
        i.created_at
      ) AS last_dt,
      TIMESTAMPDIFF(
        DAY,
        COALESCE(
          (SELECT MAX(l.created_at) FROM interaction_logs l WHERE l.interaction_id = i.id),
          i.last_activity_at,
          i.updated_at,
          i.created_at
        ),
        NOW()
      ) AS gap_days
    FROM interactions i
    LEFT JOIN clients c ON c.id = i.client_id
    LEFT JOIN users   u ON u.id = i.owner_user_id
    WHERE i.status <> 'finalizado'
      AND (i.alerts_snooze_until IS NULL OR i.alerts_snooze_until < NOW())
      AND TIMESTAMPDIFF(
            DAY,
            COALESCE(
              (SELECT MAX(l.created_at) FROM interaction_logs l WHERE l.interaction_id = i.id),
              i.last_activity_at,
              i.updated_at,
              i.created_at
            ),
            NOW()
          ) >= ?
          {$whereOwner}
    ORDER BY gap_days DESC, last_dt ASC
    LIMIT 300
  ", $params, true) ?: [];
} catch (Throwable $e) {
  $rows = [];
}

/* Mensagens via querystring */
$ok  = isset($_GET['ok'])  ? $_GET['ok']  : null;
$err = isset($_GET['err']) ? $_GET['err'] : null;

include __DIR__ . '/../../../layout/header.php';
include __DIR__ . '/../../../layout/nav.php';
?>
<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0"><i class="bi bi-bell me-2 text-warning"></i>Alertas de Inatividade</h1>
    <a class="btn btn-outline-secondary" href="<?=$BASE?>/dashboard"><i class="bi bi-arrow-left"></i> Voltar</a>
  </div>

  <?php if ($ok): ?>
    <div class="alert alert-success alert-dismissible fade show">
      <i class="bi bi-check-circle me-2"></i><?= htmlspecialchars($ok) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>
  <?php if ($err): ?>
    <div class="alert alert-danger alert-dismissible fade show">
      <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($err) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <div class="alert alert-warning">
    Exibindo atendimentos com <strong>inatividade do corretor</strong> maior ou igual a
    <strong><?= (int)$days ?></strong> <?= $labelDias ?> 
    (<?= $scopeV==='all' ? 'visão de equipe' : 'somente seus' ?>).
    <div class="small mt-1">Esse prazo é definido em <em>Admin &rarr; Configurações</em>.</div>
  </div>

  <div class="card">
    <div class="table-responsive">
      <table class="table table-dark table-hover align-middle mb-0">
        <thead>
          <tr>
            <th style="width:80px">ID</th>
            <th>Cliente</th>
            <th style="width:200px">Responsável</th>
            <th style="width:160px">Status</th>
            <th style="width:190px">Última atividade</th>
            <th style="width:160px">Gap (dias)</th>
            <th style="width:300px" class="text-end">Ações</th>
          </tr>
        </thead>
        <tbody>
        <?php foreach ($rows as $r): 
              $isOwnRow   = in_array($me_id, [(int)($r['owner_user_id'] ?? 0), (int)($r['created_by'] ?? 0)], true);
              $canAssume  = $canUpd; // qualquer um com update pode "assumir"
              $canReassign= $canUpd && ($scUpd === 'all' || rbac_can('admin','access'));
              $snoozeInfo = !empty($r['alerts_snooze_until']) ? ('<div class="small text-warning">Silenciado até '.date('d/m/Y H:i', strtotime($r['alerts_snooze_until'])).'</div>') : '';
        ?>
          <tr>
            <td>#<?= (int)$r['id'] ?></td>
            <td>
              <strong><?= htmlspecialchars($r['cliente_nome'] ?? '—') ?></strong>
              <?= $snoozeInfo ?>
            </td>
            <td><?= htmlspecialchars($r['owner_nome'] ?? '—') ?></td>
            <td>
              <span class="badge bg-<?= ['novo'=>'secondary','em_atendimento'=>'info','negociacao'=>'warning','finalizado'=>'success'][$r['status']] ?? 'secondary' ?>">
                <?= htmlspecialchars(ucfirst(str_replace('_',' ',$r['status']))) ?>
              </span>
            </td>
            <td><?= $r['last_dt'] ? date('d/m/Y H:i', strtotime($r['last_dt'])) : '—' ?></td>
            <td><?= (int)$r['gap_days'] ?></td>
            <td class="text-end">
              <div class="d-inline-flex gap-1 flex-wrap justify-content-end align-items-stretch">
                <!-- 1) VER -->
                <a class="btn btn-sm btn-outline-light" href="<?=$BASE?>/atendimentos/visualizar?id=<?=$r['id']?>" title="Ver atendimento">
                  <i class="bi bi-eye"></i>
                </a>

                <?php if ($canUpd): ?>
                  <!-- 2) NOTIFICAR por e-mail (manual via actions.php) -->
                  <form action="<?=$BASE?>/admin/alertas/actions.php" method="post" class="d-inline">
                    <input type="hidden" name="csrf" value="<?=$CSRF?>">
                    <input type="hidden" name="action" value="notificar_email">
                    <input type="hidden" name="id" value="<?=$r['id']?>">
                    <button class="btn btn-sm btn-outline-warning" title="Notificar corretor por e-mail">
                      <i class="bi bi-megaphone"></i>
                    </button>
                  </form>

                  <!-- 3) ASSUMIR -->
                  <?php if ($canAssume): ?>
                  <form action="<?=$BASE?>/admin/alertas/actions.php" method="post" class="d-inline">
                    <input type="hidden" name="csrf" value="<?=$CSRF?>">
                    <input type="hidden" name="action" value="assumir">
                    <input type="hidden" name="id" value="<?=$r['id']?>">
                    <button class="btn btn-sm btn-outline-success" title="Assumir atendimento">
                      <i class="bi bi-person-check"></i>
                    </button>
                  </form>
                  <?php endif; ?>

                  <!-- Reatribuir -->
                  <?php if ($canReassign): ?>
                  <form action="<?=$BASE?>/admin/alertas/actions.php" method="post" class="d-inline">
                    <input type="hidden" name="csrf" value="<?=$CSRF?>">
                    <input type="hidden" name="action" value="reatribuir">
                    <input type="hidden" name="id" value="<?=$r['id']?>">
                    <div class="input-group input-group-sm" style="width:160px;">
                      <select name="to_user_id" class="form-select form-select-sm" required>
                        <option value="">Reatribuir...</option>
                        <?php foreach ($owners as $o): ?>
                          <option value="<?=$o['id']?>"><?= htmlspecialchars($o['nome']) ?></option>
                        <?php endforeach; ?>
                      </select>
                      <button class="btn btn-outline-primary" title="Confirmar reatribuição">
                        <i class="bi bi-arrow-left-right"></i>
                      </button>
                    </div>
                  </form>
                  <?php endif; ?>

                  <!-- Soneca -->
                  <form action="<?=$BASE?>/admin/alertas/actions.php" method="post" class="d-inline">
                    <input type="hidden" name="csrf" value="<?=$CSRF?>">
                    <input type="hidden" name="action" value="soneca">
                    <input type="hidden" name="id" value="<?=$r['id']?>">
                    <div class="input-group input-group-sm" style="width:160px;">
                      <select name="snooze_days" class="form-select form-select-sm">
                        <option value="1">Soneca 1 dia</option>
                        <option value="2">Soneca 2 dias</option>
                        <option value="3" selected>Soneca 3 dias</option>
                        <option value="7">Soneca 7 dias</option>
                      </select>
                      <button class="btn btn-outline-light" title="Silenciar alerta">
                        <i class="bi bi-alarm"></i>
                      </button>
                    </div>
                  </form>
                <?php endif; ?>
              </div>
            </td>
          </tr>
        <?php endforeach; if (!$rows): ?>
          <tr><td colspan="7" class="text-center text-muted py-4">Sem alertas no momento.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../../../layout/footer.php'; ?>