<?php
// Diagnóstico (desative em produção)
error_reporting(E_ALL);
ini_set('display_errors', 1);

if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

rbac_require('imoveis','update');

$BASE = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$page_title = 'Fotos do Imóvel';

$property_id = (int)($_GET['id'] ?? 0);
if ($property_id <= 0) { header("Location: {$BASE}/imoveis"); exit; }

// === Carrega imóvel com fallback se 'owner_user_id' não existir ===
$imovel = null;
try {
    $imovel = fetch_one("SELECT id, titulo, owner_user_id FROM properties WHERE id = ?", [$property_id], true);
} catch (Throwable $e) {
    // Tenta sem owner_user_id
    try {
        $imovel = fetch_one("SELECT id, titulo FROM properties WHERE id = ?", [$property_id], true);
        if ($imovel) { $imovel['owner_user_id'] = null; }
    } catch (Throwable $e2) {
        http_response_code(500);
        exit('Erro ao carregar imóvel.');
    }
}
if (!$imovel) { header("Location: {$BASE}/imoveis"); exit; }

// === Escopo de update (own/all) só se a coluna existir ===
$scopeUpd = rbac_scope('imoveis','update'); // 'none' | 'own' | 'all'
$me_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
if (
    $scopeUpd === 'own'
    && array_key_exists('owner_user_id', $imovel)
    && isset($imovel['owner_user_id'])
    && (int)$imovel['owner_user_id'] !== $me_id
) {
    http_response_code(403);
    exit('Você não tem permissão para gerenciar as fotos deste imóvel.');
}

$mensagem = null;
$tipo_mensagem = 'info';

// Pastas
$upload_dir_fs  = __DIR__ . '/../../uploads/imoveis/' . $property_id . '/';           // filesystem
$upload_dir_web = rtrim($BASE, '/') . '/uploads/imoveis/' . $property_id . '/';       // web

// === POST: Upload / principal / excluir ===
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Upload
    if (isset($_FILES['fotos']) && !empty($_FILES['fotos']['name'][0])) {
        try {
            if (!is_dir($upload_dir_fs)) {
                if (!@mkdir($upload_dir_fs, 0755, true) && !is_dir($upload_dir_fs)) {
                    throw new Exception('Não foi possível criar diretório de upload.');
                }
            }

            $allowed_ext = ['jpg','jpeg','png','webp'];
            $ok = 0; $erros = [];

            foreach ($_FILES['fotos']['name'] as $k => $name) {
                if ($_FILES['fotos']['error'][$k] !== UPLOAD_ERR_OK) { $erros[] = "Falha: {$name}"; continue; }
                $tmp  = $_FILES['fotos']['tmp_name'][$k];
                $size = (int)$_FILES['fotos']['size'][$k];
                $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));

                if (!in_array($ext, $allowed_ext, true)) { $erros[] = "Extensão não permitida: {$name}"; continue; }
                if ($size > 5 * 1024 * 1024) { $erros[] = "Arquivo muito grande (5MB): {$name}"; continue; }

                $filename = uniqid('', true) . '.' . $ext;
                $dest_fs  = $upload_dir_fs . $filename;
                $dest_web = $upload_dir_web . $filename;

                if (!move_uploaded_file($tmp, $dest_fs)) { $erros[] = "Erro ao mover: {$name}"; continue; }

                // Inserção em property_photos com quedas de compatibilidade
                $insert_ok = false;
                try {
                    execute(
                        "INSERT INTO property_photos (property_id, filename, path, is_main, ordem) VALUES (?, ?, ?, 0, 0)",
                        [$property_id, $filename, $dest_web]
                    );
                    $insert_ok = true;
                } catch (Throwable $e1) {
                    try {
                        execute(
                            "INSERT INTO property_photos (property_id, filename, is_main, ordem) VALUES (?, ?, 0, 0)",
                            [$property_id, $filename]
                        );
                        $insert_ok = true;
                    } catch (Throwable $e2) {
                        try {
                            execute(
                                "INSERT INTO property_photos (property_id, filename) VALUES (?, ?)",
                                [$property_id, $filename]
                            );
                            $insert_ok = true;
                        } catch (Throwable $e3) {
                            // Reverte arquivo
                            if (file_exists($dest_fs)) @unlink($dest_fs);
                            $erros[] = "Erro ao salvar no banco: {$name}";
                        }
                    }
                }

                if ($insert_ok) { $ok++; }
            }

            if ($ok > 0) {
                $mensagem = "{$ok} foto(s) enviada(s) com sucesso!";
                $tipo_mensagem = 'success';
            }
            if (!empty($erros)) {
                $mensagem .= ($mensagem ? '<br>' : '') . "<small class='text-danger'>"
                          . implode('<br>', array_map('htmlspecialchars', $erros)) . "</small>";
                if (!$ok) $tipo_mensagem = 'danger';
            }
        } catch (Throwable $e) {
            $mensagem = "Erro ao processar upload.";
            $tipo_mensagem = 'danger';
        }
    }

    // Definir principal
    if (isset($_POST['definir_principal']) && is_numeric($_POST['definir_principal'])) {
        $foto_id = (int)$_POST['definir_principal'];
        try {
            // Se 'is_main' não existir, essas queries vão falhar e só ignoramos a ação
            execute("UPDATE property_photos SET is_main = 0 WHERE property_id = ?", [$property_id]);
            execute("UPDATE property_photos SET is_main = 1 WHERE id = ? AND property_id = ?", [$foto_id, $property_id]);
            $mensagem = "Foto principal definida com sucesso!";
            $tipo_mensagem = 'success';
        } catch (Throwable $e) {
            $mensagem = "Seu banco não possui a coluna 'is_main'. Ação indisponível.";
            $tipo_mensagem = 'warning';
        }
    }

    // Excluir foto
    if (isset($_POST['excluir_foto']) && is_numeric($_POST['excluir_foto'])) {
        $foto_id = (int)$_POST['excluir_foto'];
        try {
            $foto = fetch_one("SELECT filename, path FROM property_photos WHERE id = ? AND property_id = ?", [$foto_id, $property_id], true);
            if ($foto) {
                $file_fs = $upload_dir_fs . $foto['filename'];
                if (file_exists($file_fs)) @unlink($file_fs);
                execute("DELETE FROM property_photos WHERE id = ? AND property_id = ?", [$foto_id, $property_id]);
                $mensagem = "Foto excluída com sucesso!";
                $tipo_mensagem = 'success';
            }
        } catch (Throwable $e) {
            $mensagem = "Erro ao excluir foto.";
            $tipo_mensagem = 'danger';
        }
    }
}

// === Carregar fotos com fallback se 'path'/'ordem' não existirem ===
$fotos = [];
$hasPath = true;
try {
    $fotos = fetch_all(
        "SELECT id, filename, path, is_main, created_at FROM property_photos WHERE property_id = ? ORDER BY ordem ASC, created_at ASC",
        [$property_id],
        true
    );
} catch (Throwable $e) {
    $hasPath = false;
    // Fallback sem 'path' e sem 'ordem'
    try {
        $fotos = fetch_all(
            "SELECT id, filename, is_main, created_at FROM property_photos WHERE property_id = ? ORDER BY id ASC",
            [$property_id],
            true
        );
    } catch (Throwable $e2) {
        $fotos = [];
    }
}

$total_fotos = is_array($fotos) ? count($fotos) : 0;
$foto_principal = null;
foreach ($fotos as $f) { if (!empty($f['is_main']) && (int)$f['is_main'] === 1) { $foto_principal = $f; break; } }

// Resolve URL da imagem
$foto_web = function(array $f) use ($upload_dir_web, $hasPath) {
    if ($hasPath && !empty($f['path'])) return $f['path'];
    return $upload_dir_web . $f['filename'];
};

include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
?>
<div class="container mt-4 mb-5">
  <div class="d-flex justify-content-between align-items-center mb-4">
      <div>
          <h1 class="fw-bold mb-1">
              <i class="bi bi-images text-primary me-2"></i>
              Fotos - <?= htmlspecialchars($imovel['titulo']) ?>
          </h1>
          <p class="text-muted mb-0">Gerencie as fotos do imóvel (<?= $total_fotos ?> foto<?= $total_fotos != 1 ? 's' : '' ?>)</p>
      </div>
      <div class="d-flex gap-2">
          <a href="<?= $BASE ?>/imoveis" class="btn btn-outline-secondary">
              <i class="bi bi-list me-2"></i>Ver Imóveis
          </a>
          <a href="<?= $BASE ?>/imoveis/<?= (int)$imovel['id'] ?>/editar" class="btn btn-outline-primary">
              <i class="bi bi-pencil me-2"></i>Editar Imóvel
          </a>
      </div>
  </div>

  <?php if (!empty($mensagem)): ?>
    <div class="alert alert-<?= htmlspecialchars($tipo_mensagem) ?> alert-dismissible fade show" role="alert">
      <?= $mensagem ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
  <?php endif; ?>

  <!-- Upload -->
  <div class="card mb-4">
    <div class="card-header">
      <h5 class="mb-0"><i class="bi bi-cloud-upload me-2"></i>Adicionar Novas Fotos</h5>
    </div>
    <div class="card-body">
      <form method="POST" enctype="multipart/form-data">
        <div class="mb-3">
          <label class="form-label">Selecionar Fotos (Máx 5MB cada, JPG/PNG/WEBP)</label>
          <input type="file" name="fotos[]" class="form-control" multiple accept="image/*" required>
          <div class="form-text">Você pode selecionar múltiplas fotos de uma vez.</div>
        </div>
        <button type="submit" class="btn btn-primary">
          <i class="bi bi-upload me-2"></i>Enviar Fotos
        </button>
      </form>
    </div>
  </div>

  <!-- Status Principal -->
  <div class="alert alert-info mb-4">
    <div class="d-flex align-items-center">
      <i class="bi bi-star-fill text-warning me-2 fs-5"></i>
      <div>
        <strong>Foto Principal:</strong>
        <?php if ($foto_principal): ?>
          Definida
        <?php else: ?>
          <span class="text-warning">Nenhuma foto principal definida</span>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- Galeria -->
  <?php if ($total_fotos > 0): ?>
    <div class="row g-3">
      <?php foreach ($fotos as $idx => $f): ?>
        <div class="col-md-3 col-sm-6 mb-3">
          <div class="card h-100 position-relative">
            <img src="<?= htmlspecialchars($foto_web($f)) ?>"
                 class="card-img-top"
                 style="height:200px; object-fit:cover;"
                 alt="Foto <?= $idx+1 ?>"
                 onerror="this.src='<?= $BASE ?>/assets/img/no-image.jpg'">

            <?php if (!empty($f['is_main']) && (int)$f['is_main'] === 1): ?>
              <span class="position-absolute top-0 start-0 badge bg-warning rounded-pill m-2">
                <i class="bi bi-star-fill me-1"></i>Principal
              </span>
            <?php endif; ?>

            <div class="card-body d-flex flex-column">
              <h6 class="card-title mb-2">Foto <?= $idx+1 ?></h6>
              <small class="text-muted mb-2 flex-grow-1">
                <?= !empty($f['created_at']) ? date('d/m/Y H:i', strtotime($f['created_at'])) : '' ?>
              </small>

              <form method="POST" class="d-inline">
                <input type="hidden" name="definir_principal" value="<?= (int)$f['id'] ?>">
                <button type="submit" class="btn btn-sm btn-outline-warning w-100 mb-1" <?= (!empty($f['is_main']) && (int)$f['is_main'] === 1) ? 'disabled' : '' ?>>
                  <i class="bi bi-star<?= (!empty($f['is_main']) && (int)$f['is_main'] === 1) ? '-fill' : '' ?> me-1"></i>
                  <?= (!empty($f['is_main']) && (int)$f['is_main'] === 1) ? 'Principal' : 'Definir Principal' ?>
                </button>
              </form>

              <form method="POST" class="d-inline" onsubmit="return confirm('Excluir esta foto?');">
                <input type="hidden" name="excluir_foto" value="<?= (int)$f['id'] ?>">
                <button type="submit" class="btn btn-sm btn-outline-danger w-100">
                  <i class="bi bi-trash me-1"></i>Excluir
                </button>
              </form>
            </div>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
  <?php else: ?>
    <div class="text-center py-5">
      <i class="bi bi-images display-1 text-muted mb-3"></i>
      <h4 class="text-muted">Nenhuma foto cadastrada</h4>
      <p class="text-muted">Use a seção acima para adicionar fotos ao imóvel.</p>
    </div>
  <?php endif; ?>
</div>

<?php include __DIR__ . '/../../layout/footer.php'; ?>