<?php
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }
if (empty($_SESSION['logged_in'])) { header('Location: /troyacrm/login'); exit; }

require_once __DIR__ . '/../../app/db.php';
require_once __DIR__ . '/../../app/rbac.php';

/* === RBAC: exigir permissão para EXCLUIR imóveis === */
rbac_require('imoveis','delete');

$BASE     = defined('BASE_URL') ? BASE_URL : '/troyacrm';
$redirect = isset($_GET['redirect']) && $_GET['redirect'] ? $_GET['redirect'] : "{$BASE}/imoveis";
$id       = (int)($_GET['id'] ?? $_POST['id'] ?? 0);
if ($id <= 0) { header("Location: {$redirect}"); exit; }

/* ===== Buscar imóvel e validar escopo ===== */
try {
    $imovel = fetch_one("SELECT id, titulo, owner_user_id FROM properties WHERE id = ?", [$id], true);
    if (!$imovel) { header("Location: {$redirect}"); exit; }
} catch (Throwable $e) {
    http_response_code(500); exit('Erro ao carregar imóvel.');
}

$scope = rbac_scope('imoveis','delete'); // 'none' | 'own' | 'all'
if ($scope === 'own') {
    $me = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
    if (isset($imovel['owner_user_id']) && (int)$imovel['owner_user_id'] !== $me) {
        http_response_code(403); exit('Você não tem permissão para excluir este imóvel.');
    }
}

/* ===== CSRF helpers simples ===== */
function csrf_token_get() {
    if (function_exists('csrf_field')) { return null; } // layout pode gerar
    if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
    return $_SESSION['csrf'];
}
function csrf_check_local($token) {
    if (function_exists('csrf_check')) { return csrf_check(); }
    return isset($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], (string)$token);
}

/* ===== POST: confirmar exclusão ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $ok = csrf_check_local($_POST['csrf'] ?? '');
    if (!$ok) { http_response_code(400); exit('CSRF inválido.'); }

    try {
        // registra log antes de excluir
        $user_id = (int)($_SESSION['user']['id'] ?? $_SESSION['user_id'] ?? 0);
        execute(
            "INSERT INTO property_logs (property_id, user_id, acao, detalhes) VALUES (?, ?, 'excluido', ?)",
            [$imovel['id'], $user_id, 'Exclusão do imóvel "'.$imovel['titulo'].'"']
        );

        // exclui imóvel
        execute("DELETE FROM properties WHERE id = ?", [$imovel['id']]);

        header("Location: {$redirect}?deleted=1");
        exit;
    } catch (Throwable $e) {
        http_response_code(500); exit('Erro ao excluir imóvel.');
    }
}

/* ===== GET: página de confirmação ===== */
$page_title = 'Excluir imóvel';
include __DIR__ . '/../../layout/header.php';
include __DIR__ . '/../../layout/nav.php';
$csrf = csrf_token_get();
?>
<div class="container my-4">
  <div class="card border-danger">
    <div class="card-header bg-danger text-white">
      <i class="bi bi-trash"></i> Confirmar exclusão
    </div>
    <div class="card-body">
      <p>Tem certeza de que deseja excluir o imóvel <strong><?= htmlspecialchars($imovel['titulo']) ?></strong> (ID <?= (int)$imovel['id'] ?>)?</p>
      <div class="alert alert-warning">
        <i class="bi bi-exclamation-triangle"></i> Esta ação não pode ser desfeita.
      </div>
      <form method="post" class="d-flex gap-2">
        <input type="hidden" name="id" value="<?= (int)$imovel['id'] ?>">
        <?php if ($csrf): ?>
          <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
        <?php elseif (function_exists('csrf_field')): ?>
          <?= csrf_field(); ?>
        <?php endif; ?>
        <a href="<?= htmlspecialchars($redirect) ?>" class="btn btn-secondary">
          <i class="bi bi-arrow-left"></i> Cancelar
        </a>
        <button class="btn btn-danger">
          <i class="bi bi-trash"></i> Excluir definitivamente
        </button>
      </form>
    </div>
  </div>
</div>
<?php include __DIR__ . '/../../layout/footer.php'; ?>