<?php

function getFeaturedProperties(PDO $pdo, int $limit = 4): array
{
    $sql = "SELECT * FROM properties WHERE status = 'publicado' ORDER BY created_at DESC LIMIT :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function searchProperties(PDO $pdo, array $filters = [], int $limit = 20): array
{
    $where  = ["status = 'publicado'"];
    $params = [];

    if (!empty($filters['tipo'])) {
        $where[]             = "tipo = :tipo";
        $params[':tipo']     = $filters['tipo'];
    }

    if (!empty($filters['finalidade'])) {
        $where[]                 = "finalidade = :finalidade";
        $params[':finalidade']   = $filters['finalidade'];
    }

    if (!empty($filters['cidade'])) {
        $where[]               = "cidade LIKE :cidade";
        $params[':cidade']     = '%' . $filters['cidade'] . '%';
    }

    $sql = "SELECT * FROM properties";
    if ($where) {
        $sql .= " WHERE " . implode(' AND ', $where);
    }
    $sql .= " ORDER BY created_at DESC LIMIT :limit";

    $stmt = $pdo->prepare($sql);

    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);

    $stmt->execute();
    return $stmt->fetchAll();
}

function getPropertyById(PDO $pdo, int $id): ?array
{
    $sql  = "SELECT * FROM properties WHERE id = :id AND status = 'publicado' LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    $row = $stmt->fetch();
    return $row ?: null;
}

function getPropertyMainPhoto(PDO $pdo, int $propertyId): ?string
{
    $sql  = "SELECT path
               FROM property_photos
              WHERE property_id = :id
              ORDER BY is_main DESC, ordem ASC, id ASC
              LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $propertyId, PDO::PARAM_INT);
    $stmt->execute();
    $row = $stmt->fetch();
    return $row['path'] ?? null;
}

function getPropertyPhotos(PDO $pdo, int $propertyId): array
{
    $sql  = "SELECT *
               FROM property_photos
              WHERE property_id = :id
              ORDER BY is_main DESC, ordem ASC, id ASC";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $propertyId, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function getPropertyFeatures(PDO $pdo, int $propertyId, int $limit = 12): array
{
    $sql = "SELECT pf.nome, pf.icone
              FROM property_features pf
              INNER JOIN property_feature_values pfv ON pfv.feature_id = pf.id
             WHERE pfv.property_id = :property_id
               AND (pf.categoria = 'imovel' OR pf.categoria = 'geral')
             ORDER BY pf.nome ASC
             LIMIT :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':property_id', $propertyId, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function formatPrice(float $price): string
{
    return 'R$ ' . number_format($price, 0, ',', '.');
}
