<?php

function getFeaturedProperties(PDO $pdo, int $limit = 4): array
{
    $sql = "SELECT * FROM properties WHERE status = 'publicado' ORDER BY created_at DESC LIMIT :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Imóveis para o HERO da home.
 *
 * Neste momento está trazendo os últimos imóveis publicados.
 * Se o seu CRM tiver uma coluna específica para "principal" ou "destaque",
 * você pode ajustar o WHERE, por exemplo:
 *
 *   WHERE status = 'publicado' AND principal = 1
 *
 */
function getHeroProperties(PDO $pdo, int $limit = 5): array
{
    $sql = "SELECT * FROM properties WHERE status = 'publicado' ORDER BY created_at DESC LIMIT :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Busca imóveis com filtros.
 *
 * Filtros aceitos em $filters:
 * - 'tipo'         => string
 * - 'finalidade'   => 'venda' | 'aluguel' (ou o que vier do CRM)
 * - 'cidade'       => string (usa LIKE)
 * - 'condominio'   => string (nome do condomínio gravado em properties.condominio)
 * - 'preco_min'    => numérico (>=)
 * - 'preco_max'    => numérico (<=)
 */
function searchProperties(PDO $pdo, array $filters = [], int $limit = 20): array
{
    $where  = ["status = 'publicado'"];
    $params = [];

    if (!empty($filters['tipo'])) {
        $where[]         = "tipo = :tipo";
        $params[':tipo'] = $filters['tipo'];
    }

    if (!empty($filters['finalidade'])) {
        $where[]               = "finalidade = :finalidade";
        $params[':finalidade'] = $filters['finalidade'];
    }

    if (!empty($filters['cidade'])) {
        $where[]           = "cidade LIKE :cidade";
        $params[':cidade'] = '%' . $filters['cidade'] . '%';
    }

    // Filtro por condomínio (nome salvo em properties.condominio)
    if (!empty($filters['condominio'])) {
        $where[]               = "condominio = :condominio";
        $params[':condominio'] = $filters['condominio'];
    }

    // Faixa de preço mínima
    if (!empty($filters['preco_min']) && is_numeric($filters['preco_min'])) {
        $where[]              = "preco >= :preco_min";
        $params[':preco_min'] = (float) $filters['preco_min'];
    }

    // Faixa de preço máxima
    if (!empty($filters['preco_max']) && is_numeric($filters['preco_max'])) {
        $where[]              = "preco <= :preco_max";
        $params[':preco_max'] = (float) $filters['preco_max'];
    }

    $sql = "SELECT * FROM properties";
    if ($where) {
        $sql .= " WHERE " . implode(' AND ', $where);
    }
    $sql .= " ORDER BY created_at DESC LIMIT :limit";

    $stmt = $pdo->prepare($sql);

    foreach ($params as $key => $value) {
        // sem tipo explícito para não ter problema com decimal em preço
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);

    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Lista de condomínios para montar selects.
 * Usa a tabela `condominios` (apenas ativos).
 *
 * Retorna um array de linhas com:
 * - id
 * - nome
 * - cidade
 * - uf
 */
function getCondominios(PDO $pdo): array
{
    $sql = "
        SELECT id, nome, cidade, uf
          FROM condominios
         WHERE ativo = 1
         ORDER BY nome ASC
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    return $stmt->fetchAll();
}

function getPropertyById(PDO $pdo, int $id): ?array
{
    $sql  = "SELECT * FROM properties WHERE id = :id AND status = 'publicado' LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    $row = $stmt->fetch();
    return $row ?: null;
}

function getPropertyMainPhoto(PDO $pdo, int $propertyId): ?string
{
    $sql  = "SELECT path
               FROM property_photos
              WHERE property_id = :id
              ORDER BY is_main DESC, ordem ASC, id ASC
              LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $propertyId, PDO::PARAM_INT);
    $stmt->execute();
    $row = $stmt->fetch();
    return $row['path'] ?? null;
}

function getPropertyPhotos(PDO $pdo, int $propertyId): array
{
    $sql  = "SELECT *
               FROM property_photos
              WHERE property_id = :id
              ORDER BY is_main DESC, ordem ASC, id ASC";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $propertyId, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function getPropertyFeatures(PDO $pdo, int $propertyId, int $limit = 12): array
{
    $sql = "SELECT pf.nome, pf.icone
              FROM property_features pf
              INNER JOIN property_feature_values pfv ON pfv.feature_id = pf.id
             WHERE pfv.property_id = :property_id
               AND (pf.categoria = 'imovel' OR pf.categoria = 'geral')
             ORDER BY pf.nome ASC
             LIMIT :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':property_id', $propertyId, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function formatPrice(float $price): string
{
    return 'R$ ' . number_format($price, 0, ',', '.');
}