<?php
require __DIR__ . '/inc/db.php';
require __DIR__ . '/inc/functions.php';

$id = isset($_GET['id']) ? (int) $_GET['id'] : 0;

if ($id <= 0) {
    http_response_code(404);
    $property = null;
} else {
    $property = getPropertyById($pdo, $id);
}

if (!$property) {
    $pageTitle   = "Imóvel não encontrado";
    $currentPage = "";
    include __DIR__ . '/inc/header.php';
    ?>
    <section class="page-hero">
        <div class="container page-hero-content">
            <div>
                <p class="hero-tag">IMÓVEL INDISPONÍVEL</p>
                <h1>Ops, não encontramos esse imóvel</h1>
                <p class="hero-subtitle">
                    Ele pode ter sido removido, estar reservado ou o link pode estar incorreto.
                </p>
            </div>
        </div>
    </section>

    <section class="section-contact">
        <div class="container">
            <p style="margin-bottom: 1.5rem;">
                Se você estava interessado em um imóvel específico, fale com nossa equipe para encontrar
                alternativas semelhantes.
            </p>
            <a href="imoveis.php" class="btn btn-outline">Voltar para a lista de imóveis</a>
        </div>
    </section>
    <?php
    include __DIR__ . '/inc/footer.php';
    exit;
}

$pageTitle   = $property['titulo'] . " - Detalhes do imóvel";
$currentPage = "";

// Fotos e características
$photos   = getPropertyPhotos($pdo, (int) $property['id']);
$features = getPropertyFeatures($pdo, (int) $property['id'], 24);

// Array simples só com URLs das fotos para o JS e para a galeria
$photoUrls = [];
foreach ($photos as $p) {
    if (!empty($p['path'])) {
        $photoUrls[] = $p['path'];
    }
}

// Foto principal = primeira foto válida
$photoMain = $photoUrls[0] ?? null;

$bairro  = trim((string) ($property['bairro'] ?? ''));
$cidade  = trim((string) ($property['cidade'] ?? ''));
$uf      = trim((string) ($property['uf'] ?? ''));

$local = $bairro;
if ($cidade) {
    $local = $local ? $local . ' • ' . $cidade : $cidade;
}
if ($uf) {
    $local = $local ? $local . ' / ' . $uf : $uf;
}

$badgeText  = ($property['finalidade'] ?? '') === 'aluguel' ? 'Para alugar' : 'À venda';
$badgeClass = ($property['finalidade'] ?? '') === 'aluguel' ? 'badge-rent' : '';

include __DIR__ . '/inc/header.php';
?>

<!-- Marca d'água nas fotos (principal + lightbox) -->
<style>
/* garante que os contêineres aceitem pseudo-elemento posicionado */
.gallery-main,
.lightbox-image {
    position: relative;
}

/* aplica a logo como marca d’água usando o arquivo na raiz de /troya9 */
.gallery-main::after,
.lightbox-image::after {
    content: "";
    position: absolute;
    right: 16px;
    bottom: 16px;
    width: 320px;          /* largura fixa da logo */
    height: 120px;          /* altura aproximada (ajuste conforme proporção da logo) */
    background-image: url('/troya9/logo-troya.png');
    background-repeat: no-repeat;
    background-position: right bottom;
    background-size: contain;
    opacity: 0.65;         /* mais forte pra ficar bem visível */
    filter: grayscale(100%);
    pointer-events: none;
    z-index: 5;            /* acima da imagem de fundo e abaixo das setas */
}

/* se ainda existir alguma marca d’água antiga em outro CSS, dá pra “matar” aqui.
   Exemplo (ajuste o seletor real se existir):
   .old-watermark-class { display:none !important; }
*/
</style>

<section class="property-hero">
    <div class="container property-hero-content">
        <div>
            <p class="hero-tag">IMÓVEL EM DESTAQUE</p>
            <h1><?php echo htmlspecialchars($property['titulo']); ?></h1>
            <?php if ($local): ?>
                <p class="property-location-line"><?php echo htmlspecialchars($local); ?></p>
            <?php endif; ?>
            <div class="property-tags">
                <span class="pill <?php echo $badgeClass; ?>">
                    <?php echo htmlspecialchars($badgeText); ?>
                </span>
                <?php if (!empty($property['tipo'])): ?>
                    <span class="pill pill-outline">
                        <?php echo htmlspecialchars(ucfirst($property['tipo'])); ?>
                    </span>
                <?php endif; ?>
            </div>
        </div>

        <div class="property-price-block">
            <p class="property-price-big">
                <?php echo formatPrice((float) $property['preco']); ?>
            </p>
            <p class="property-price-note">
                Valor sujeito a alteração sem aviso prévio.
            </p>
            <a href="contato.php" class="btn btn-primary full-width">Quero saber mais</a>
        </div>
    </div>
</section>

<section class="property-layout">
    <div class="container property-layout-grid">
        <div class="property-gallery">
            <?php
            $styleMain = '';
            if ($photoMain) {
                $styleMain = " style=\"background-image: url('" . htmlspecialchars($photoMain, ENT_QUOTES, 'UTF-8') . "');\"";
            }
            ?>
            <div class="gallery-main"<?php echo $styleMain; ?>></div>

            <?php if (count($photoUrls) > 1): ?>
                <div class="gallery-thumbs">
                    <?php
                    // limita a quantidade de thumbs exibidas (melhor para performance)
                    $maxThumbs = 16;
                    foreach ($photoUrls as $index => $url):
                        if ($index >= $maxThumbs) {
                            break;
                        }
                    ?>
                        <div class="gallery-thumb"
                             data-index="<?php echo (int)$index; ?>"
                             data-bg="<?php echo htmlspecialchars($url, ENT_QUOTES, 'UTF-8'); ?>"></div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <aside class="property-sidebar">
            <div class="info-block">
                <h2>Resumo do imóvel</h2>
                <div class="info-grid">
                    <?php if (!empty($property['quartos'])): ?>
                        <div class="info-item">
                            <span class="info-label">Quartos</span>
                            <span class="info-value"><?php echo (int) $property['quartos']; ?></span>
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($property['suites'])): ?>
                        <div class="info-item">
                            <span class="info-label">Suítes</span>
                            <span class="info-value"><?php echo (int) $property['suites']; ?></span>
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($property['banheiros'])): ?>
                        <div class="info-item">
                            <span class="info-label">Banheiros</span>
                            <span class="info-value"><?php echo (int) $property['banheiros']; ?></span>
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($property['vagas'])): ?>
                        <div class="info-item">
                            <span class="info-label">Vagas</span>
                            <span class="info-value"><?php echo (int) $property['vagas']; ?></span>
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($property['area_construida_m2'])): ?>
                        <div class="info-item">
                            <span class="info-label">Área construída</span>
                            <span class="info-value">
                                <?php echo (float) $property['area_construida_m2']; ?> m²
                            </span>
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($property['area_lote_m2'])): ?>
                        <div class="info-item">
                            <span class="info-label">Área do terreno</span>
                            <span class="info-value">
                                <?php echo (float) $property['area_lote_m2']; ?> m²
                            </span>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="info-block">
                <h2>Localização</h2>
                <p class="info-text">
                    <?php if (!empty($property['endereco'])): ?>
                        <?php echo htmlspecialchars($property['endereco']); ?><br>
                    <?php endif; ?>

                    <?php if ($bairro): ?>
                        <?php echo htmlspecialchars($bairro); ?> •
                    <?php endif; ?>

                    <?php if ($cidade): ?>
                        <?php echo htmlspecialchars($cidade); ?>
                    <?php endif; ?>

                    <?php if ($uf): ?>
                        / <?php echo htmlspecialchars($uf); ?>
                    <?php endif; ?>

                    <?php if (!empty($property['cep'])): ?>
                        <br>CEP <?php echo htmlspecialchars($property['cep']); ?>
                    <?php endif; ?>
                </p>
            </div>

            <div class="info-block">
                <h2>Próximos passos</h2>
                <p class="info-text">
                    Ficou interessado neste imóvel? Entre em contato com nossa equipe para agendar uma visita
                    ou tirar suas dúvidas sobre condições comerciais.
                </p>
                <a href="contato.php" class="btn btn-outline full-width">Falar com um consultor</a>
            </div>
        </aside>
    </div>
</section>

<?php if (!empty($property['descricao'])): ?>
<section class="property-description">
    <div class="container">
        <h2>Descrição do imóvel</h2>
        <p><?php echo nl2br(htmlspecialchars($property['descricao'])); ?></p>
    </div>
</section>
<?php endif; ?>

<?php if ($features): ?>
<section class="property-features">
    <div class="container">
        <h2>Características</h2>
        <div class="feature-chips">
            <?php foreach ($features as $feature): ?>
                <span class="feature-chip">
                    <?php echo htmlspecialchars($feature['nome']); ?>
                </span>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php
// Lightbox / carrossel de fotos
$jsPhotoArray = json_encode($photoUrls, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
?>

<div class="lightbox" id="photo-lightbox">
    <div class="lightbox-backdrop"></div>
    <div class="lightbox-inner">
        <button type="button" class="lightbox-close" aria-label="Fechar">&times;</button>

        <!-- seta voltar -->
        <button type="button" class="lightbox-arrow lightbox-prev" aria-label="Imagem anterior">
            &#10094;
        </button>

        <div class="lightbox-image"></div>

        <!-- seta avançar -->
        <button type="button" class="lightbox-arrow lightbox-next" aria-label="Próxima imagem">
            &#10095;
        </button>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const photoUrls = <?php echo $jsPhotoArray ?: '[]'; ?>;
    if (!photoUrls.length) return;

    const lightbox  = document.getElementById('photo-lightbox');
    const imageDiv  = lightbox.querySelector('.lightbox-image');
    const btnClose  = lightbox.querySelector('.lightbox-close');
    const btnPrev   = lightbox.querySelector('.lightbox-prev');
    const btnNext   = lightbox.querySelector('.lightbox-next');
    const backdrop  = lightbox.querySelector('.lightbox-backdrop');
    const main      = document.querySelector('.gallery-main');
    const thumbs    = document.querySelectorAll('.gallery-thumb');

    let currentIndex = 0;

    // --- PRELOAD DE IMAGENS GRANDES ---
    const preloaded = {};

    function preloadImage(index) {
        if (index < 0 || index >= photoUrls.length) return;
        if (preloaded[index]) return;

        const img = new Image();
        img.src = photoUrls[index];
        preloaded[index] = img;
    }

    function preloadAround(index) {
        preloadImage(index);       // atual
        preloadImage(index + 1);   // próxima
        preloadImage(index - 1);   // anterior
    }

    // já deixa algumas em cache assim que a página carrega
    for (let i = 0; i < Math.min(5, photoUrls.length); i++) {
        preloadImage(i);
    }

    function showImage(index) {
        if (!photoUrls.length) return;

        if (index < 0) index = photoUrls.length - 1;
        if (index >= photoUrls.length) index = 0;
        currentIndex = index;

        const url = photoUrls[currentIndex] || '';
        const safeUrl = url.replace(/'/g, "\\'");
        imageDiv.style.backgroundImage = "url('" + safeUrl + "')";

        // garante que atual / próxima / anterior já estejam carregadas
        preloadAround(currentIndex);
    }

    function openLightbox(index) {
        if (!photoUrls.length) return;
        showImage(index);
        lightbox.classList.add('open');
        document.body.style.overflow = 'hidden';
    }

    function closeLightbox() {
        lightbox.classList.remove('open');
        document.body.style.overflow = '';
    }

    if (btnClose) {
        btnClose.addEventListener('click', closeLightbox);
    }

    if (backdrop) {
        backdrop.addEventListener('click', closeLightbox);
    }

    if (btnNext) {
        btnNext.addEventListener('click', function () {
            showImage(currentIndex + 1);
        });
    }

    if (btnPrev) {
        btnPrev.addEventListener('click', function () {
            showImage(currentIndex - 1);
        });
    }

    // clique na foto principal abre o lightbox
    if (main) {
        main.style.cursor = 'pointer';
        main.addEventListener('click', function () {
            openLightbox(currentIndex);
        });
    }

    // Lazy load das miniaturas (thumbs)
    let thumbObserver = null;

    if ('IntersectionObserver' in window) {
        thumbObserver = new IntersectionObserver(function (entries, observer) {
            entries.forEach(function (entry) {
                if (entry.isIntersecting) {
                    const el = entry.target;
                    const bg = el.getAttribute('data-bg');
                    if (bg) {
                        el.style.backgroundImage =
                            "url('" + bg.replace(/'/g, "\\'") + "')";
                        el.removeAttribute('data-bg');
                    }
                    observer.unobserve(el);
                }
            });
        }, { rootMargin: '100px' });
    }

    // thumbs trocam a foto principal e usam lazy load
    thumbs.forEach(function (thumb) {
        thumb.style.cursor = 'pointer';

        const idxAttr = thumb.getAttribute('data-index');
        const thumbIndex = idxAttr ? parseInt(idxAttr, 10) : 0;

        if (thumbObserver) {
            thumbObserver.observe(thumb);
        } else {
            // fallback: carrega todas de uma vez se não houver IntersectionObserver
            const bg = thumb.getAttribute('data-bg');
            if (bg) {
                thumb.style.backgroundImage =
                    "url('" + bg.replace(/'/g, "\\'") + "')";
                thumb.removeAttribute('data-bg');
            }
        }

        thumb.addEventListener('click', function () {
            currentIndex = thumbIndex;

            // troca a foto principal
            if (main && photoUrls[currentIndex]) {
                main.style.backgroundImage =
                    "url('" + photoUrls[currentIndex].replace(/'/g, "\\'") + "')";
            }

            // garante que as próximas já venham em cache
            preloadAround(currentIndex);
        });
    });

    // teclado
    document.addEventListener('keydown', function (e) {
        if (!lightbox.classList.contains('open')) return;

        if (e.key === 'Escape') {
            closeLightbox();
        } else if (e.key === 'ArrowRight') {
            showImage(currentIndex + 1);
        } else if (e.key === 'ArrowLeft') {
            showImage(currentIndex - 1);
        }
    });
});
</script>

<?php
include __DIR__ . '/inc/footer.php';